<?php

use function ZWE\count_safe;

/**
 * WRTeam
 *
 * @author     Hendrik Heneke <hendrik@pixelwunder.net>
 */
class WRTeam extends BaseWRTeam
{

	private $__presets;

	public function addAssignment(User $user)
	{
		// prüfen, ob bereits ein Einsatz für diesen User vorliegt
		foreach ($this->Assignments as $a) {
			if ($a->User->id == $user->id) {
				return $a;
			}
		}
		// falls nicht, hinzufügen
		$assignment = new Assignment();
		$assignment->User = $user;
		$assignment->WRTeam = $this;
		$this->Assignments->add($assignment);
		$assignment->save();
		$this->Event->invalidateAssignmentplans();
		return $assignment;
	}

	public function addReplaceCompetition(Competition $competition)
	{
		$competition->WRTeam = $this;
		if (!$competition->id > 0) {
			$this->Competitions->add($competition);
		}
		$competition->save();
		return $competition;
	}

	public function getAssignments($state)
	{
		$query = Doctrine_Query::create()
			->select('a.*, u.*, c.*')
			->from('Assignment a')
			->innerJoin('a.User u')
			->innerJoin('u.Club c')
			->where('a.team_id = ?', $this->id)
			->orderBy('u.surname, u.name');
		if ($state != NULL) {
			$query->andWhereIn('a.state', $state);
		}
		return $query->execute();
	}

	public function getCompetitions()
	{
		$query = Doctrine_Query::create()
			->select('c.*')
			->from('Competition c')
			->where('c.team_id = ?', $this->id)
			->orderBy('c.sortorder');
		return $query->execute();
	}

	/**
	 * Gibt die höchste erforderliche Lizenzpriorität für das WR-Team zurück
	 * für die angegebene Turnierart zurück. Gibt License::PRIO_MAX_VALUE zurück,
	 * wenn die Turnierart nicht im WR-Team vorkommt.
	 * @param mixed $dancetype Turnierart: Dancetype-Objekt oder dessen ID, default null
	 * @return int höchste erforderliche Lizenzpriorität
	 */
	public function getHighestLicensePriorityRequired($dancetype = null)
	{
		$startclass = $this->getStartclassWithHighestLicensePriority($dancetype);
		if ($startclass != null) {
			return $startclass->License->prio;
		} else {
			return License::PRIO_MAX_VALUE;
		}
	}

	/**
	 * Gibt die Startklasse mit der höchsten Lizenzpriorität für die angegebene
	 * Turnierart zurück. Wird keine Turnierart angeben, wird die höchste Lizenzpriorität
	 * aller Turnierarten gesucht.
	 * @param mixed $dancetype Turnierart: Dancetype-Objekt oder dessen ID, default null
	 * @return Startclass Startklasse mit der höchsten Lizenzpriorität oder null, falls
	 * die Turnierart nicht im WR-Team vorkommt
	 */
	public function getStartclassWithHighestLicensePriority_OLD($dancetype = null)
	{
		$query = Doctrine_Query::create()
			->select('s.*, l.*')
			->from('Startclass s')
			->innerJoin('s.License l')
			->innerJoin('s.Competitions c')
			->innerJoin('c.WRTeam t')
			->where('t.id = ?', $this->id)
			->orderBy('l.prio ASC')
			->limit(1);
		if ($dancetype != null) {
			$dancetype_id = $dancetype; // if given as id
			if ($dancetype instanceof Dancetype) {
				$dancetype_id = $dancetype->id; // if given as object
			}
			$query->addWhere('c.art_id = ?', $dancetype_id);
		}
		return $query->fetchOne();
	}

	public function getStartclassWithHighestLicensePriority($dancetype = null)
	{
		$dancetype_id = null;
		if ($dancetype != null) {
			if ($dancetype instanceof Dancetype) {
				$dancetype_id = $dancetype->id;
			} else {
				$dancetype_id = $dancetype;
			}
		}

		$highestStartclass = null;
		$highestPriority = License::PRIO_MAX_VALUE;
		foreach ($this->Competitions as $competition) {
			$competitionStartclass = null;

			if ($dancetype_id == null || $competition->Dancetype->id == $dancetype_id) {
				$competitionStartclass = $competition->Startclass;
			}

			if ($competitionStartclass != null) {
				if ($highestPriority > $competitionStartclass->License->prio) {
					$highestPriority = $competitionStartclass->License->prio;
					$highestStartclass = $competitionStartclass;
				}
			}
		}
		return $highestStartclass;
	}

	public function getStartclassWithHighestLicensePriorityName()
	{
		$s = $this->getStartclassWithHighestLicensePriority();
		if ($s != null) {
			return $s->name;
		}
		return '';
	}

	public function getEinsatzCount($states = Assignment::STATE_DEFAULT)
	{
		$query = Doctrine_Query::create()
			->select('COUNT(a.id) AS _einsatz_count')
			->from('Assignment a')
			->innerJoin('a.WRTeam t')
			->where('t.id = ?')
			->andWhereIn('a.state', $states);
		return $query->fetchOne(array($this->id))->_einsatz_count;
	}

	public function getCompetitionCount()
	{
		$query = Doctrine_Query::create()
			->select('COUNT(c.id) AS _competition_count')
			->from('Competition c')
			->innerJoin('c.WRTeam t')
			->where('t.id = ?');
		return $query->fetchOne(array($this->id))->_competition_count;
	}

	public function hasDanceType($dancetype)
	{
		$id = 0;
		if ($dancetype instanceof Dancetype) {
			$id = $dancetype->id;
		} else {
			$id = $dancetype;
		}
		$query = Doctrine_Query::create()
			->select('COUNT(c.id) AS _dancetype_count')
			->from('WRTeam t')
			->innerJoin('t.Competitions c')
			->where('t.id = ?', $this->id)
			->andWhere('c.art_id = ?', $id);
		return $query->fetchOne()->_dancetype_count > 0;
	}

	public function isComplete()
	{
		$confirmed = $this->getAssignments(Assignment::STATE_CONFIRMED);
		return count_safe($confirmed) >= $this->judgecount;
	}

	public function isInvited()
	{
		$invited = $this->getAssignments(array(Assignment::STATE_INVITED, Assignment::STATE_CONFIRMED));
		return count_safe($invited) >= $this->judgecount;
	}

	public function removeAssignment(Assignment $assignment)
	{
		foreach ($this->Assignments as $i => $a) {
			if ($assignment->state != Assignment::STATE_DEFAULT) {
				throw new Exception('Removing Assignments is only allowed with STATE_DEFAULT!');
			}
			if ($assignment->id == $a->id) {
				$this->Assignments->remove($i);
				$assignment->delete();
			}
		}
		$this->Event->invalidateAssignmentplans();
	}

	public function removeCompetition(Competition $competition)
	{
		foreach ($this->Competitions as $i => $c) {
			if ($competition->id == $c->id) {
				$this->Competitions->remove($i);
				$competition->delete();
			}
		}
	}

	public function setPresets(array $presets)
	{
		$this->_set('serialized_preset', json_encode($presets));
		$this->__presets = $presets;
	}

	public function setPreset($preset, $value)
	{
		if ($this->__presets == null) {
			$this->__presets = $this->unserializePresets();
		}
		if ($this->__presets == null) {
			$this->__presets = array();
		}
		$this->__presets[$preset] = $value;
		$this->_set('serialized_preset', json_encode($this->__presets));
	}

	public function getPreset($preset)
	{
		if ($this->__presets == null) {
			$this->__presets = $this->unserializePresets();
		}
		if (isset($this->__presets[$preset])) {
			return $this->__presets[$preset];
		} // defaults zurueckgeben
		else {
			switch ($preset) {
				case "q_freigabe";
					return \ZWE\Settings\SettingsRegistry::getInstance()->get('zwe_preset_availability')->getMandatorValue($this->Event->mandator_id);
				case "q_mandator";
					return \ZWE\Settings\SettingsRegistry::getInstance()->get('zwe_preset_mandator')->getMandatorValue($this->Event->mandator_id);
				case "q_herkunft":
					return \ZWE\Settings\SettingsRegistry::getInstance()->get('zwe_preset_origin')->getMandatorValue($this->Event->mandator_id);
			}
		}
		return null;
	}

	private function unserializePresets() {
		$raw = $this->_get('serialized_preset');
		if ($raw !== null && $raw !== '') {
			$decodedJson = @json_decode($raw, true);
			if (json_last_error() === JSON_ERROR_NONE) {
				return $decodedJson;
			} else {
				return unserialize($raw);
			}
		} else {
			return null;
		}
	}

	public function updateDates()
	{
		$minDateFrom = array_reduce($this->Competitions->getData(), function ($prev, $cur) {
			if (strcmp($cur->date_from, $prev) < 0) {
				$prev = $cur->date_from;
			}
			return $prev;
		}, '9999-12-31');

		$maxDateTo = array_reduce($this->Competitions->getData(), function ($prev, $cur) {
			if ($cur->date_to) {
				if (strcmp($cur->date_to, $prev) > 0) {
					$prev = $cur->date_to;
				}
			}
			return $prev;
		}, '1900-01-01');

		$this->date_from = $minDateFrom;
		$this->date_to = $maxDateTo === '1900-01-01' ? null : $maxDateTo;
	}

	public function getCompetitionDates(): array
	{
		if (!$this->date_from) {
			return [];
		}

		if ($this->date_to) {
			return \ZWE\calculateAllDatesInRange($this->getDateTimeObject('date_from'), $this->getDateTimeObject('date_to'));
		} else {
			return \ZWE\calculateAllDatesInRange($this->getDateTimeObject('date_from'));
		}
	}

	public function formatDate($format = 'd.m.Y'): string
	{
		if ($this->date_to) {
			return "{$this->getDateTimeObject('date_from')->format($format)} – {$this->getDateTimeObject('date_to')->format($format)}";
		} else {
			return $this->getDateTimeObject('date_from')->format($format);
		}
	}

	public function getDisplayTitle(): string
	{
		if ($this->date_from) {
			$infix = $this->date_from ? 'vom' : 'am';
			return "{$this->name} {$infix} {$this->formatDate()}";
		} else {
			return $this->name;
		}
	}

	public function isNewPlanningMode(): bool
	{
		return $this->getPreset('q_freigabe') == -1;
	}
}
