<?php

/**
 * User
 *
 * @author     Hendrik Heneke <hendrik@pixelwunder.net>
 * @property-read int|null $__stat_f
 * @property-read int|null $__stat_e
 * @property-read int|null $__stat_i
 * @property-read int|null $__stat_z
 * @property-read int|null $__stat_a
 */
class User extends BaseUser
{

	const STATE_REGISTERED = 3;
	const STATE_VALIDATED = 2;
	const STATE_ACTIVE = 1;
	const STATE_UNLICENSED = -2;
	const STATE_INACTIVE = -1;
	const STATE_DELETED = -20;

	const MALE = 'm';
	const FEMALE = 'w';

	private $__stat_props = array('__stat_f', '__stat_e', '__stat_i', '__stat_z', '__stat_a');
	private $__avcache;

	/**
	 * @var \Tanzsport\ESV\API\Model\Funktionaer\Funktionaer
	 */
	private $_esv_data;

	static function allStates()
	{
		return array(
			"3" => 'registriert',
			"2" => 'validiert',
			"1" => 'freigeschaltet',
			"-2" => 'unlizensiert',
			"-1" => 'deaktiviert',
		);
	}

	static function allStatesSystemUsers()
	{
		return array(
			"1" => 'aktiviert',
			"-1" => 'deaktiviert',
		);
	}

	/**
	 * Überprüft das Passwort des Benutzers.
	 * @param string $password Passwort (Klartext)
	 * @return bool
	 */
	public function checkPassword($password)
	{
		return $this->passwd == md5($password);
	}

	/**
	 * Gibt den Einsatz mit der angegebenen ID und dem angegeben Status zurück. Wird keine Status angegeben,
	 * wird der Einsatz unabhängig vom Status zurückgegeben. Gibt null zurück, falls der Einsatz nicht zum
	 * Benutzer gehört und der Status nicht übereinstimmt.
	 * @param int $id Einsatz-ID
	 * @param int $state Einsatz-Status (fakultativ)
	 * @return Assignment
	 */
	public function findAssignment($id, $state = null)
	{
		foreach ($this->Assignments as $assignment) {
			if ($state !== null) {
				if ($assignment->id == $id && $assignment->state == $state) {
					return $assignment;
				}
			} else {
				if ($assignment->id == $id) {
					return $assignment;
				}
			}
		}
		return null;
	}

	/**
	 * Lädt den Benutzer aus der Session. Deprecated: Auf den Benutzer
	 * im aktuellen Kontext soll nur noch über dem UserContext
	 * zugegriffen werden.
	 *
	 * @deprecated
	 * @return User Session-Benutzer
	 */
	public static function fromSession()
	{
		return \ZWE\User\UserContext::getInstance()->getUser();
	}

	/**
	 * Gibt die deutsche Anrese zurück.
	 * @return string
	 */
	public function getAnrede()
	{
		switch ($this->gender) {
			case self::FEMALE:
				return 'Frau';
			case self::MALE:
			default:
				return 'Herr';
			// @codeCoverageIgnoreStart
		}
		// @codeCoverageIgnoreEnd
	}

	public function getAvailabilities($from, $to)
	{
		return Doctrine::getTable('Availability')
			->find('by-user-between_dates', array($this->id, $from, $to));
	}

	/**
	 * Gibt alle zukünftigen Einsätze mit Status STATE_CONFIRMED zurück.
	 * @return Doctrine_Collection
	 */
	public function getConfirmedAssignments()
	{
		return $this->getAssignments(Assignment::STATE_CONFIRMED);
	}

	/**
	 * Gibt alle zukünftigen Einsätze mit Status STATE_INVITED zurück.
	 * @return Doctrine_Collection
	 */
	public function getInvitations()
	{
		return $this->getAssignments(Assignment::STATE_INVITED);
	}

	/**
	 * Gibt alle Einsätze mit dem angegebenen Status zurück.
	 * @param int|int[] $state Einsatz-Status
	 * @param bool|null $infuture true = nur zukünftige Einsätze, false = nur vergangene Einsätze, null = alle Einsätze
	 * @param str|null $sort Sortierung (ASC, DESC)
	 * @param int|null $offset Offset
	 * @param int|null $limit Limit
	 * @return Doctrine_Collection
	 */
	public function getAssignments($state = null, $infuture = true, $sort = 'ASC', $offset = null, $limit = null)
	{
		$query = Doctrine_Query::create()
			->select('a.id, a.state, a.invited, a.confirmed, a.cancelled')
			->from('Assignment a')
			->innerJoin('a.WRTeam t')
			->innerJoin('t.Event e')
			->where('a.user_id = ?', $this->id)
			->orderBy('e.datum');
		if ($state !== null) {
			if (is_array($state)) {
				$query->andWhereIn('a.state', $state);
			} else {
				$query->andWhere('a.state = ?', $state);
			}
		}
		if ($infuture !== null) {
			$now = new DateTime();
			if ($infuture) {
				$query->andWhere('IFNULL(t.date_to, t.date_from) >= ?', $now->format('Y-m-d'));
			} else {
				$query->andWhere('IFNULL(t.date_to, t.date_from) < ?', $now->format('Y-m-d'));
			}
		}
		if ($sort !== null) {
			$query->orderBy("e.datum {$sort}");
		}
		if ($offset !== null) {
			$query->offset(intval($offset));
		}
		if ($limit !== null) {
			$query->limit(intval($limit));
		}
		return $query->execute();
	}

	/**
	 * @param Mandator $mandator
	 * @return string
	 */
	public function getMandatorGrantsAsText(Mandator $mandator)
	{
		foreach ($this->AvailableMandators as $m) {
			if ($m->id == $mandator->id) {
				$lr = $m->isOperatedByUserWithACL($this, true, null, null, null) ? 'L' : '-';
				$lw = $m->isOperatedByUserWithACL($this, null, null, true, null) ? 'S' : '-';
				$pr = $m->isOperatedByUserWithACL($this, null, true, null, null) ? 'L' : '-';
				$pw = $m->isOperatedByUserWithACL($this, null, null, null, true) ? 'S' : '-';
				return '[V:' . $lr . $lw . '|P:' . $pr . $pw . ']';
			}
		}
		return '[V:--|P:--]';
	}

	/**
	 * @param $admin_read
	 * @param $plan_read
	 * @param $admin_write
	 * @param $plan_write
	 * @return array
	 */
	public function getMandators($admin_read, $plan_read, $admin_write, $plan_write)
	{
		$ret = array();
		foreach ($this->AvailableMandators as $m) {
			if ($m->isOperatedByUserWithACL($this, $admin_read, $plan_read,
				$admin_write, $plan_write)
			) {
				$ret[] = $m;
			}
		}
		return $ret;
	}

	public function getStatistics()
	{
		$stat = new stdClass();
		$stat->free = 0;
		$stat->assigned = 0;
		$stat->invited = 0;
		$stat->confirmed = 0;
		$stat->cancelled = 0;

		$set = true;
		foreach ($this->__stat_props as $prop) {
			if (!isset($this->$prop)) {
				$set = false;
				break;
			}
		}

		if (!$set) {
			$query = Doctrine_Query::create()
				->select('a.state, COUNT(a.id) AS __count, a.state AS __state')
				->from('Assignment a')
				->innerJoin('a.User u')
				->where('u.id = ?', $this->id)
				->groupBy('a.state');
			$res = $query->execute();
			foreach ($res as $record) {
				switch ($record->__state) {
					case Assignment::STATE_DEFAULT:
						$stat->assigned = $record->__count;
						break;
					case Assignment::STATE_INVITED:
						$stat->invited = $record->__count;
						break;
					case Assignment::STATE_CONFIRMED:
						$stat->confirmed = $record->__count;
						break;
					case Assignment::STATE_CANCELLED:
						$stat->cancelled = $record->__count;
						break;
				}
			}

			$query = Doctrine_Query::create()
				->select('COUNT(a.id) AS __count')
				->from('Availability a')
				->innerJoin('a.User u')
				->where('u.id = ?', $this->id);
			$res = $query->fetchOne();
			$stat->free = $res->__count;
		} else {
			$stat->free = $this->__stat_f;
			$stat->assigned = $this->__stat_e;
			$stat->invited = $this->__stat_i;
			$stat->confirmed = $this->__stat_z;
			$stat->cancelled = $this->__stat_a;
		}
		return $stat;
	}

	public static function getStatusAsTextStatic($state)
	{
		$states = self::allStates();
		return isset($states[$state]) ? $states[$state] : 'n.def.';
	}

	/**
	 * Gibt die Mandanten-Vorauswahl bei ZWE-Benutzern zurück nach
	 * vorheriger Prüfung zurück ($DefaultMandator).
	 *
	 * @return Mandator
	 * @since 2.5
	 */
	public function getPreselectedMandator()
	{
		if ($this->DefaultMandator != null) {
			if ($this->DefaultMandator->isOperatedBy($this)) {
				return $this->DefaultMandator;
			}
		}
		return null;
	}

	/**
	 * Gewährt eine Benutzerrolle
	 * @param Role $role Benutzerrolle
	 */
	public function grant(Role $role)
	{
		if (!$this->isGranted($role)) {
			$this->Roles->add($role);
		}
	}

	/**
	 * Revoziert eine Benutzerrolle
	 * @param Role $role Benutzerrolle
	 */
	public function revoke(Role $role)
	{
		if ($this->isGranted($role)) {
			foreach ($this->Roles as $idx => $r) {
				if ($r->id == $role->id) {
					$this->Roles->remove($idx);
				}
			}
		}
	}

	/**
	 * Gibt zurück, ob der Benutzer für das angegebene Datum freigegeben hat
	 * @param DateTime $dateTime Datum
	 * @return bool
	 */
	public function isAvailable(DateTime $dateTime)
	{
		foreach ($this->Availabilities as $av) {
			if ($av->getDateTimeObject('datum') == $dateTime) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Gibt zurück, ob ein WR-Kandidat freigegeben hat.
	 * Kann nur einen korrekten Wert liefern, wenn der WR als Kandidat geladen wurde (via WRTeamRepository).
	 *
	 * @return bool
	 */
	public function candidateIsAvailable()
	{
		return $this->get('__candidate_available') > 0;
	}

	/**
	 * Gibt zurück, ob der Benutzer einen Einsatz am angegebenen Datum hat
	 * (eingesetzt, eingeladen oder zugesagt).
	 *
	 * @param DateTime $dateTime Datum
	 * @return bool
	 */
	public function hasAssignments(DateTime $dateTime)
	{
		foreach ($this->Assignments as $ass) {
			if ($ass->WRTeam->Event->getDateTimeObject('datum') == $dateTime && ($ass->isDefault() || $ass->isInvited() || $ass->isConfirmed())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Gibt zurück, ob ein WR-Kandidat konkurrierende Einsätze (eingesetzt, eingeladen, zugesagt) hat.
	 * Kann nur einen korrkten Wert liefern, wenn der WR als Kandidat geladen wurde (via WRTeamRepository).
	 *
	 * @return bool
	 */
	public function candidateHasAssignments()
	{
		return $this->get('__candidate_assignments') > 0;
	}

	/**
	 * Gibt zurück, ob der Benutzer einen abgesagten Einsatz am angegebenen
	 * Datum hat.
	 *
	 * @param DateTime $dateTime Datum
	 * @return bool
	 */
	public function hasCancellations(DateTime $dateTime)
	{
		foreach ($this->Assignments as $ass) {
			if ($ass->WRTeam->Event->getDateTimeObject('datum') ==
				$dateTime && $ass->isCancelled()
			) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Gibt zurück, ob ein WR-Kandidat konkurrierende Absagen hat.
	 * Kann nur einen korrkten Wert liefern, wenn der WR als Kandidat geladen wurde (via WRTeamRepository).
	 *
	 * @return bool
	 */
	public function candidateHasCancellations()
	{
		return $this->get('__candidate_cancellations') > 0;
	}

	/**
	 * Gibt zurück, ob das Benutzerkonto aktiv ist.
	 * @return bool
	 */
	public function isActive()
	{
		return $this->active == self::STATE_ACTIVE;
	}

	/**
	 * Gibt zurück, ob das Benutzerkonto unlizensiert ist.
	 * @return bool
	 */
	public function isUnlicensed()
	{
		return $this->active == self::STATE_UNLICENSED;
	}

	public function mayLogin()
	{
		return $this->isActive() || $this->isUnlicensed();
	}

	private function isGranted($role)
	{
		$rolename = $role;
		if (is_a($role, "Role")) {
			$rolename = $role->role;
		}
		foreach ($this->Roles as $r) {
			if ($r->role == $rolename) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Gibt zurück, ob der Benutzer ein Admin-Benutzer ist, also der Rolle ROLE_ADMIN zugeordnet ist.
	 * @return bool
	 */
	public function isAdmin()
	{
		return $this->isGranted(Role::ADMIN);
	}

	/**
	 * Gibt zurück, ob der Benutzer ein Admin- oder ZWE-Benutzer ist, als den Rollen ROLE_ADMIN und/oder
	 * ROLE_ZWE zugeordnet ist.
	 * @return bool
	 */
	public function isAdminOrZWE()
	{
		return $this->isAdmin() || $this->isZWE();
	}

	/**
	 * Gibt zurück, ob der Benutzer ein WR-Benutzer ist, also der Rolle ROLE_WR zugeordnet ist.
	 * @return bool
	 */
	public function isWR()
	{
		return $this->isGranted(Role::WR);
	}

	/**
	 * Gibt zurück, ob der Benutzer ein ZWE-Benutzer ist, also der Rolle ROLE_ZWE zugeordnet ist.
	 * @return bool
	 */
	public function isZWE()
	{
		return $this->isGranted(Role::ZWE);
	}

	public function isEsvWR()
	{
		if ($this->isWR()) {
			if ($this->Club != null && $this->Club->LTV != null && $this->Club->LTV->id != 'EXT') {
				return true;
			}
		}
		return false;
	}

	/**
	 * Login-Prozedur; gibt das Benutzerobjekt bei erfolgreichem Login zurück, ansonsten null.
	 * @param string $username Benutzername
	 * @param string $password Passwort (Klartext)
	 * @return User
	 */
	public static function login($username, $password)
	{
		if (trim($username) == '') {
			return null;
		}

		$user = Doctrine::getTable('User')->findOneBy('dtvnr', $username);
		if ($user) {
			if (($user->isActive() || $user->isUnlicensed()) && $user->checkPassword($password)) {
				return $user;
			}
		}
		return null;
	}

	/**
	 * Logout-Prozedur; setzt die Zeit des letzten Logins, resetted die Anzahl der erfolglosen
	 * Login-Versuche und entfernt das Benutzer-Objekt aus der Session.
	 * @return void
	 */
	public function logout()
	{
		$this->setDateTimeObject('lastlogin', new DateTime());
		$this->fails = 0;
		$this->save();
		$this->removeFromContext();
	}

	/**
	 * Entfernt das Benutzer-Objekt aus dem Kontext.
	 */
	public function removeFromContext()
	{
		\ZWE\User\UserContext::getInstance()->clear();
	}

	/**
	 * Setzt die Mandanten-Vorauswahl bei ZWE-Benutzern zurück.
	 */
	public function resetPreselectedMandator()
	{
		$this->DefaultMandator = null;
	}

	/**
	 * Setzt den Passwort-Hash (MD5).
	 * @param $value Passwort (Klartext)
	 */
	public function setPassword($value)
	{
		$this->_set('passwd', md5($value));
	}

	/**
	 * Setzt die Vorauswahl des Mandanten bei ZWE-Benutzern nach
	 * vorheriger Prüfung ($DefaultMandator). Bei Benutzern ohne die
	 * Rollenzuordnung ROLE_ZWE wird das Attribut auf null gesetzt.
	 *
	 * @param Mandator $mandator Standard-Mandant für ZWE-Benutzer
	 * @since 2.5
	 */
	public function setPreselectedMandator(Mandator $mandator)
	{
		if ($this->isZWE()) {
			if ($mandator->isOperatedBy($this)) {
				$this->DefaultMandator = $mandator;
			} else {
				$this->DefaultMandator = null;
			}
		} else {
			$this->DefaultMandator = null;
		}
	}

	/**
	 * Speichert das Benutzer-Objekt als Kontext-Variable.
	 */
	public function saveInContext()
	{
		\ZWE\User\UserContext::getInstance()->setUser($this);
	}

	/**
	 * Aktualisiert die effektive Lizenz bei Wertungsrichter-Benutzern.
	 */
	public function updateEffectiveLicenses()
	{
		if ($this->isWR()) {

			// ACHTUNG: kleinster Wert - höchste Priorität

			// 1. eingeschränkte startklassen

			// startklasse eingeschränkt nur wenn ungleich null und prio kleiner als PRIO_MAX_VALUE
			if ($this->LicenseEffective != null && $this->LicenseEffective->prio != License::PRIO_MAX_VALUE) {
				// lizenz nur dann einschränken, wenn wert prio (effektiv) > prio (lizenz)

				// standard
				if ($this->LicenseEffective->prio > $this->LicenseSt->prio) {
					$this->LicenseStEffective = $this->LicenseEffective;
				} else {
					$this->LicenseStEffective = $this->LicenseSt;
				}

				// latein
				if ($this->LicenseEffective->prio > $this->LicenseLat->prio) {
					$this->LicenseLatEffective = $this->LicenseEffective;
				} else {
					$this->LicenseLatEffective = $this->LicenseLat;
				}
			} // startklasse nicht eingeschränkt, eingebene Lizenzen direkt übernehmen
			else {
				$this->LicenseStEffective = $this->LicenseSt;
				$this->LicenseLatEffective = $this->LicenseLat;
			}

			// 2. kombination

			// effektive kombi-lizenz entspricht der effektiven lizenz st oder lat mit dem höheren prio-wert
			if ($this->LicenseStEffective->prio >= $this->LicenseLatEffective->prio) {
				$this->LicenseKombiEffective = $this->LicenseStEffective;
			} else {
				$this->LicenseKombiEffective = $this->LicenseLatEffective;
			}
		} else {
			$this->LicenseSt = null;
			$this->LicenseStEffective = null;
			$this->LicenseLat = null;
			$this->LicenseLatEffective = null;
			$this->LicenseEffective = null;
			$this->LicenseKombiEffective = null;
		}
	}

	/**
	 * Schließt den Validierungsprozess der E-Mailadresse ab, setzt den Status auf STATE_VALIDATED
	 * und speichert die Änderungen in der Datenbank.
	 * @return void
	 */
	public function validationComplete()
	{
		$this->setDateTimeObject('validated', new DateTime());
		$this->activation_code = null;
		$this->active = self::STATE_VALIDATED;
		$this->save();
	}

	public function isMale()
	{
		return $this->gender == self::MALE ? true : false;
	}

	public function isFemale()
	{
		return $this->gender == self::FEMALE ? true : false;
	}

	public function isLicenseSufficient(WRTeam $team)
	{
		$danceTypes = array(Dancetype::KOMBI, Dancetype::LATEIN, Dancetype::STANDARD);

		$sufficient = true;

		foreach ($danceTypes as $danceType) {
			$startClass = $team->getStartclassWithHighestLicensePriority($danceType);
			if ($startClass != null) {
				switch ($danceType) {
					case Dancetype::KOMBI:
						if ($startClass->License->prio < $this->LicenseKombiEffective->prio) {
							$sufficient = false;
						}
						break;
					case Dancetype::LATEIN:
						if ($startClass->License->prio < $this->LicenseLatEffective->prio) {
							$sufficient = false;
						}
						break;
					case Dancetype::STANDARD:
						if ($startClass->License->prio < $this->LicenseStEffective->prio) {
							$sufficient = false;
						}
						break;
				}
			}
		}

		return $sufficient;
	}

	public function hasEsvDaten()
	{
		return $this->esv_check && $this->esv_data;
	}

	public function getEsvDaten()
	{
		if (!$this->_esv_data) {
			if ($this->esv_check && $this->esv_data) {
				$this->_esv_data = unserialize($this->esv_data);
			}
		}
		return $this->_esv_data;
	}

	public function setEsvDaten(\Tanzsport\ESV\API\Model\Funktionaer\Funktionaer $esvDaten, DateTime $zeitstempel)
	{
		$this->esv_data = serialize($esvDaten);
		$this->setDateTimeObject('esv_check', $zeitstempel);
		$this->last_license_year = intval($zeitstempel->format('Y'));
		$this->_esv_data = $esvDaten;
	}

	public function isEsvNamensAbweichung()
	{
		if (!$this->isEsvWR()) {
			return false;
		}
		return $this->normalizeName($this->name) != $this->normalizeName($this->getEsvDaten()->vorname) ||
			$this->normalizeName($this->surname) != $this->normalizeName($this->getEsvDaten()->nachname);
	}

	private function normalizeName($input)
	{
		if ($input) {
			return str_replace(' ', '', trim(strtolower($input)));
		}
	}

	public function isLicenseExpired()
	{
		if ($this->Club->LTV->id != 'EXT') {
			if ($this->last_license_year === null || $this->last_license_year < date('Y')) {
				return true;
			}
		}
		return false;
	}

	public function getLastLicenseYearAsString()
	{
		return $this->last_license_year == null ? '?' : "{$this->last_license_year}";
	}
}
