<?php
namespace ZWE\User;

use ZWE\Context\StaticContextHolder;
use ZWE\Singleton;

/**
 * Benutzer-Kontext, der den Zugriff auf den aktuellen Benutzer ermöglicht
 * (sofern vorhanden).
 *
 * Abhängig von der PHP-SAPI wird der Benutzer entweder aus der HTTP-Session
 * oder aus einer statischen Variable geladen. Dafür stehen zwei
 * unterschiedliche UserContextHolder zur Verfügung.
 *
 * Bei Objekten dieser Klasse handelt es sich um Singletons, die nicht
 * geklont werden können.
 *
 * @see         UserContextHolder
 *
 * @author      Hendrik Heneke <hendrik@pixelwunder.net>
 * @since       2.5
 */
class UserContext extends Singleton
{

	private static $instance;

	private $contextHolder;

	private function  __construct()
	{
		if (strtolower(php_sapi_name()) == 'cli') {
			$this->contextHolder = new StaticContextHolder();
		} else {
			$this->contextHolder = new UserCISessionContextHolder();
		}
	}

	/**
	 * Gibt den aktuellen Benutzer-Kontext zurück. Der Kontext wird
	 * erzeugt, falls noch keine Instanz existiert.
	 *
	 * @return UserContext Benutzer-Kontext
	 */
	public static final function getInstance()
	{
		if (self::$instance == null) {
			self::$instance = new UserContext();
		}
		return self::$instance;
	}

	public function clear()
	{
		$this->contextHolder->clearContext();
	}

	/**
	 * Gibt den aktuellen Benutzer zur&uuml;ck.
	 *
	 * @return |User|null
	 */
	public final function getUser()
	{
		return $this->contextHolder->getContext();
	}

	/**
	 * Speichert den Benutzer im aktuellen Kontext.
	 *
	 * @param \User $user Benutzer
	 */
	public final function setUser(\User $user)
	{
		$this->contextHolder->setContext($user);
	}

}