<?php
namespace ZWE\Settings;

class Setting
{

	const CATEGORY_SYSTEM = 'S';
	const CATEGORY_MANDATOR = 'M';
	const CATEGORY_USER = 'U';

	const TYPE_STRING = 's';
	const TYPE_TEXT = 't';
	const TYPE_BOOLEAN = 'b';
	const TYPE_NUMERIC = 'n';
	const TYPE_LTV = 'l';
	const TYPE_OPTIONS = 'o';

	private static $def_val_required = 'trim|required|max_length[65535]';
	private static $def_val = 'trim|max_length[65535]';

	private $category;
	private $name;
	private $type;
	private $default_value;
	private $validators;

	private $dbSetting = null;

	private $initializedWithCategory = null;
	private $options = [];

	public function __construct($name, $type, $category, $default_value, $required, $addtionalValidators = '')
	{
		$this->name = $name;
		$this->type = $type;
		$this->default_value = $default_value;
		$this->category = $category;
		if ($required) {
			$this->validators = self::$def_val_required;
		} else {
			$this->validators = self::$def_val;
		}
		if ($addtionalValidators != '') {
			$this->validators .= '|' . $addtionalValidators;
		}
	}

	/**
	 * Gibt zurück, ob die Mandanten-Einstellung der System-Einstellung
	 * entspricht.
	 *
	 * @param Mandator $mandator
	 * @return bool
	 */
	public function mandatorEqualsSystemSetting(\Mandator $mandator)
	{
		return $this->getSystemValue() === $this->getMandatorValue(
			$mandator->id);
	}

	/**
	 * Gibt die Einstellungs-Kategorie (System, Mandant, Benutzer) zurück.
	 *
	 * @return string
	 */
	public function getCategory()
	{
		return $this->category;
	}

	/**
	 * Gibt den Standardwert der Einstellung zurück.
	 *
	 * @return mixed
	 */
	public function getDefaultValue()
	{
		return $this->default_value;
	}

	/**
	 * Gibt den Namen der Einstellung zurück.
	 *
	 * @return string
	 */
	public function getName()
	{
		return $this->name;
	}

	/**
	 * Gibt den Wertetyp der Einstellung zurück.
	 *
	 * @return string
	 */
	public function getType()
	{
		return $this->type;
	}

	/**
	 * Gibt die Validator-Konfiguration der Einstellung zurück (CodeIgniter-spezifisch).
	 *
	 * @return string
	 */
	public function getValidators()
	{
		return $this->validators;
	}

	/**
	 * Alias für getSystemValue().
	 *
	 * @return mixed
	 */
	public function getValue()
	{
		return $this->getSystemValue();
	}

	private function mandatorFromContextOrInput($input)
	{
		if ($input == null) {
			return \ZWE\Mandator\MandatorContext::getInstance()->getMandator()->id;
		} else {
			return $input;
		}
	}

	/**
	 * Gibt den Wert der System-Einstellung zurück. Sofern keine persistierte
	 * Einstellung gefunden wurde, wird der Standard-Wert zurückgegeben.
	 *
	 * @return mixed
	 */
	public function getSystemValue()
	{
		$this->init(self::CATEGORY_SYSTEM);
		return $this->convertInputValue($this->dbSetting->value);
	}

	/**
	 * Gibt den Wert der Mandanten-Einstellung zurück. Wenn keine persistierte
	 * Mandanten-Einstellung gefunden wurde, wird die System-Einstellung
	 * zurückgegeben.
	 *
	 * @return mixed
	 */
	public function getMandatorValue($mandator_id = null)
	{
		$this->init(self::CATEGORY_MANDATOR, $this->mandatorFromContextOrInput($mandator_id));
		return $this->convertInputValue($this->dbSetting->value);
	}

	/**
	 * Validiert ein PersistentSetting-Objekt (zur Verwendung vor dem
	 * Speichern).
	 *
	 * @param PersistentSetting $ps
	 */
	private function validate(\PersistentSetting $ps)
	{
		if ($ps->name == null) {
			throw new SettingsException('Name der Einstellung fehlt!');
		}
		if ($ps->category == self::CATEGORY_MANDATOR) {
			if ($ps->mandator_id == null) {
				throw new SettingsException('Mandanten-ID der Einstellung ' . $ps->name . ' fehlt!');
			}
		}
	}

	/**
	 * Gibt die Optionen einen Auswahl-Feldes zurück.
	 *
	 * @return array
	 */
	public function getOptions() {
		return $this->options;
	}

	/**
	 * Setzt die Optionen für ein Auswahl-Feld.
	 *
	 * @param array $options
	 * @return $this
	 */
	public function setOptions(array $options) {
		$this->options = $options;
		return $this;
	}

	/**
	 * Alias für setSystemValue().
	 *
	 * @param mixed $value
	 */
	public function setValue($value)
	{
		$this->setSystemValue($value);
	}

	/**
	 * Setzt den Wert der System-Einstellung.
	 *
	 * @param mixed $value
	 */
	public function setSystemValue($value)
	{
		$this->init(self::CATEGORY_SYSTEM);
		if ($this->convertInputValue($value) == $this->default_value) {
			if ($this->dbSetting->id != null) {
				$this->dbSetting->delete();
				$this->dbSetting = $this->initNewPersistentSystemSetting($this->default_value);
			}
		} else {
			$this->dbSetting->value = $this->convertInputValue($value);
			$this->dbSetting->save();
		}
	}

	/**
	 *
	 * @param <type> $value
	 * @param <type> $mandator_id
	 */
	public function setMandatorValue($value, $mandator_id)
	{
		$this->init(self::CATEGORY_MANDATOR, $mandator_id);
		if ($this->equalsSystemOrDefaultValue($value)) {
			if ($this->dbSetting->id != null) {
				$this->dbSetting->delete();
				$this->dbSetting = $this->initNewPersistentMandatorSettingFromSystemOrDefault($mandator_id);
			}
		} else {
			$this->dbSetting->value = $this->convertInputValue($value);
			$this->dbSetting->save();
		}
	}

	/**
	 * Initialisiert die Member-Variable $dbSetting mit den Daten
	 * aus der Datenbank. Wird keine persistente Einstellung gefunden,
	 * wird $dbSetting mit einem neuen, nicht-persistentes
	 * PersistentSetting-Objekt initialisiert.
	 *
	 * Die Initialisierungssequenz wird nur ausgeführt, wenn das Objekt
	 * nocht nicht mit der Kategorie $category initialisiert wurde.
	 *
	 * @param $category string Einstellungs-Kategorie zur Initialisierung
	 * @param $mandator_id int Mandanten-ID (null für Kategorie System)
	 */
	private function init($category, $mandator_id = null)
	{
		if ($category == self::CATEGORY_SYSTEM) {
			$this->dbSetting = $this->queryForSystemSetting();
			if ($this->dbSetting == null) {
				$this->dbSetting = $this->initNewPersistentSystemSetting($this->default_value);
			}
		} elseif ($category == self::CATEGORY_MANDATOR) {
			$this->dbSetting = $this->queryForMandatorSetting($mandator_id);
			if ($this->dbSetting == null) {
				$systemSetting = $this->queryForSystemSetting();
				if ($systemSetting != null) {
					$this->dbSetting = $this->convertToMandatorSetting($systemSetting, $mandator_id);
				} else {
					$this->dbSetting = $this->initNewPersistentMandatorSetting($mandator_id, $this->default_value);
				}
			}
		} else {
			throw new SettingsException('Kategorie ' . $category . ' wird nicht unterstützt!');
		}
		$this->initializedWithCategory = $category;
	}

	private function initNewPersistentSystemSetting($value = null)
	{
		$ps = new \PersistentSetting();
		$ps->category = self::CATEGORY_SYSTEM;
		$ps->name = $this->name;
		$ps->value = $this->convertInputValue($value);
		return $ps;
	}

	private function initNewPersistentMandatorSetting($mandator_id, $value = null)
	{
		if ($mandator_id == null) {
			throw new SettingsException('Initialisierung einer Mandanten-Einstellung ohne Mandanten-ID ist unmöglich!');
		}
		$ps = new \PersistentSetting();
		$ps->category = self::CATEGORY_MANDATOR;
		$ps->name = $this->name;
		$ps->mandator_id = $mandator_id;
		$ps->value = $this->convertInputValue($value);
		return $ps;
	}

	private function initNewPersistentMandatorSettingFromSystemOrDefault($mandator_id)
	{
		if ($mandator_id == null) {
			throw new SettingsException('Initialisierung einer Mandanten-Einstellung ohne Mandanten-ID ist unmöglich!');
		}
		$systemSetting = $this->queryForSystemSetting();
		$value = $this->default_value;
		if ($systemSetting != null) {
			$value = $systemSetting->value;
		}
		return $this->initNewPersistentMandatorSetting($mandator_id, $value);
	}

	/**
	 * Konvertiert eine System-Einstellung in eine nicht-persistente
	 * Mandanten-Einstellung.
	 *
	 * @param PersistentSetting $ps die System-Einstellung
	 * @param int $mandator_id Mandanten-ID
	 * @return PersistentSetting die nicht-persistente Mandanten-Einstellung
	 */
	private function convertToMandatorSetting(\PersistentSetting $ps, $mandator_id)
	{
		if ($this->category != self::CATEGORY_MANDATOR) {
			throw new SettingsException('Kann Einstellung ' . $ps->name . ' nicht in eine Mandanten-Einstelung konvertieren!');
		}
		if ($ps->category == self::CATEGORY_MANDATOR) {
			throw new SettingsException('Kann Mandanten-Einstellung ' . $ps->name . ' nicht in eine Einstellung eines anderen Mandanten konvertieren!');
		}
		if ($mandator_id == null) {
			throw new SettingsException('Kann System-Einstellung ohne Mandanten-ID nicht in Mandanten-Einstellung konvertieren!');
		}
		$convertedSetting = new \PersistentSetting();
		$convertedSetting->fromArray($ps->toArray());
		$convertedSetting->id = null;
		$convertedSetting->category = self::CATEGORY_MANDATOR;
		$convertedSetting->mandator_id = $mandator_id;
		return $convertedSetting;
	}

	private function queryForMandatorSetting($mandator_id)
	{
		return $this->queryForSettingWithCategory(self::CATEGORY_MANDATOR, $mandator_id);
	}

	private function queryForSystemSetting()
	{
		return $this->queryForSettingWithCategory(self::CATEGORY_SYSTEM);
	}

	private function equalsSystemOrDefaultValue($value)
	{
		$systemSetting = $this->queryForSystemSetting();
		if ($systemSetting != null) {
			return $this->convertInputValue($value) === $systemSetting->value;
		} else {
			return $this->convertInputValue($value) === $this->default_value;
		}
	}

	/**
	 * Fragt die Datenbank nach der aktuellen Einstellung mit der gegebenen
	 * Kategorie ab. Gibt null zurück, sofern keine persistente
	 * Einstellung in der gegebenen Kategorie existiert.
	 *
	 * @param $category string Kategorie
	 * @param $mandator_id int Mandanten-ID (null für Kategorie System)
	 * @return PersistentSetting
	 */
	private function queryForSettingWithCategory($category, $mandator_id = null)
	{
		if ($category == self::CATEGORY_SYSTEM && $mandator_id != null) {
			throw new SettingsException('Abfrage nach Kategorie System benötigt keine Mandanten-ID!');
		} elseif ($category == self::CATEGORY_MANDATOR && $mandator_id == null) {
			throw new SettingsException('Abfrage nach Kategorie Mandant erfordert eine Mandanten-ID!');
		} else {
			$query = \Doctrine_Query::create()
				->select('s.*')
				->from('PersistentSetting s')
				->where('s.name = ?', $this->name)
				->andWhere('s.category = ?', $category);
			if ($category == self::CATEGORY_MANDATOR) {
				$query->andWhere('s.mandator_id = ?', $mandator_id);
			}
			return $query->fetchOne();
		}
	}

	/**
	 * Wandet den Eingabe-Wert je nach Wertetyp der Einstellung um.
	 *
	 * @param $input Eingabe-Wert
	 * @return mixed
	 */
	private function convertInputValue($input)
	{
		switch ($this->type) {
			case self::TYPE_BOOLEAN:
				return $input > 0 ? TRUE : FALSE;
			case self::TYPE_NUMERIC:
				return is_numeric($input) ? $input + 0 : 0;
			case self::TYPE_STRING:
			case self::TYPE_TEXT:
			default:
				return $input;
		}
	}

}
