<?php
namespace ZWE\Repository;

use ZWE\Mandator\MandatorUtils;
use ZWE\Settings\SettingsRegistry;
use function ZWE\count_safe;

class WRTeamRepository extends AbstractRepository
{

	public function find($id)
	{
		return $this->findInternal('WRTeam', $id);
	}

	public function getAuditPager($page = 1, $results = 5, $owners = null)
	{
		return new \Doctrine_Pager($this->getAuditQuery($owners), $page, $results);
	}

	private function getAuditQuery($owners = null)
	{
		$query = \Doctrine_Query::create()
			->select('t.*, e.*')
			->addSelect('(SELECT COUNT(a1.id) FROM Assignment a1 WHERE a1.team_id = t.id AND a1.state = ' . \Assignment::STATE_INVITED . ') AS missing_confirms')
			->addSelect('(SELECT COUNT(a2.id) FROM Assignment a2 WHERE a2.team_id = t.id AND a2.state = ' . \Assignment::STATE_DEFAULT . ') AS missing_invitations')
			->addSelect('(t.judgecount - (SELECT COUNT(a3.id) FROM Assignment a3 WHERE a3.team_id = t.id AND a3.state IN (' . \Assignment::STATE_DEFAULT . ',' . \Assignment::STATE_INVITED . ', ' . \Assignment::STATE_CONFIRMED . '))) AS missing_assignments')
			->from('WRTeam t')
			->innerJoin('t.Event e')
			->where('e.datum >= NOW()')
			->having('missing_confirms > 0 OR missing_invitations > 0 OR missing_assignments > 0')
			->orderBy('e.datum, e.name');
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			$query->andWhereIn('e.mandator_id', $owner_ids);
		}
		return $query;
	}

	public function findAllInFuture($owners = null)
	{
		$query = \Doctrine_Query::create()
			->select('t.*, e.*, c.*')
			->from('WRTeam t')
			->innerJoin('t.Event e')
			->innerJoin('e.Club c')
			->where('e.datum >= NOW()')
			->orderBy('e.datum, t.name');
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			if (count_safe($owner_ids) > 0) {
				$query->andWhereIn('e.mandator_id', $owner_ids);
			}
		}
		return $query->execute();
	}

	/**
	 * @param \WRTeam $team
	 * @param bool $lizenzErzwingen
	 * @param bool $verfuegbarkeitBeruecksichtigen
	 * @param string $herkunft
	 * @param bool $doppelteClubsVermeiden
	 * @param bool $freigabeBeruecksichtigen
	 * @param null $mandanten
	 * @return \Doctrine_Collection
	 */
	public function findCandidates(\WRTeam $team, $lizenzErzwingen = true, $verfuegbarkeitBeruecksichtigen = true, $herkunft = 'all', $doppelteClubsVermeiden = true, $freigabeBeruecksichtigen = true, $mandanten = null)
	{
		$date = array_keys($team->getCompetitionDates())[0];

		$ownerLtv = SettingsRegistry::getInstance()->get('zwe_ltv')->getMandatorValue($team->Event->Owner);

		$query = \Doctrine_Query::create()
			->select('u.*, cl.*, owner.*, lkombi.*, llat.*, lst.*, team.*, e.*')
			->from('User u')
			->innerJoin('u.Roles r')
			->innerJoin('u.Club cl')
			->innerJoin('u.LicenseKombiEffective lkombi')
			->innerJoin('u.LicenseLatEffective llat')
			->innerJoin('u.LicenseStEffective lst')
			->innerJoin('cl.Owner owner');

		// cache verfügbarkeit
		$query->addSelect('(SELECT COUNT(avcache.id) FROM Availability avcache WHERE avcache.user_id = u.id AND avcache.datum = \'' . $date . '\') as __candidate_available');
		// cache anzahl konkurrierende einsätze
		$query->addSelect('(SELECT COUNT(ascache.id) FROM Assignment AS ascache INNER JOIN ascache.WRTeam asteam WHERE ascache.user_id = u.id AND ascache.state IN (0, 1, 2) and asteam.date_from = \'' . $date . '\') as __candidate_assignments');
		// cache anzahl konkurrierender absagen
		$query->addSelect('(SELECT COUNT(cacache.id) FROM Assignment AS cacache INNER JOIN cacache.WRTeam cateam WHERE cacache.user_id = u.id AND cacache.state IN (-1) and cateam.date_from = \'' . $date . '\') as __candidate_cancellations');

		// statistiken
		$query->addSelect('(SELECT COUNT(av_count.user_id) FROM Availability av_count WHERE av_count.user_id = u.id) AS __stat_f');
		$query->addSelect('(SELECT COUNT(as_e_count.user_id) FROM Assignment as_e_count WHERE as_e_count.user_id = u.id AND as_e_count.state = ' . \Assignment::STATE_DEFAULT . ') AS __stat_e');
		$query->addSelect('(SELECT COUNT(as_i_count.user_id) FROM Assignment as_i_count WHERE as_i_count.user_id = u.id AND as_i_count.state = ' . \Assignment::STATE_INVITED . ') AS __stat_i');
		$query->addSelect('(SELECT COUNT(as_z_count.user_id) FROM Assignment as_z_count WHERE as_z_count.user_id = u.id AND as_z_count.state = ' . \Assignment::STATE_CONFIRMED . ') AS __stat_z');
		$query->addSelect('(SELECT COUNT(as_a_count.user_id) FROM Assignment as_a_count WHERE as_a_count.user_id = u.id AND as_a_count.state = ' . \Assignment::STATE_CANCELLED . ') AS __stat_a');

		// nur aktive benutzer abfragen
		$query->addWhere('u.active = ?', array(\User::STATE_ACTIVE));

		// nur wr-benutzer abfragen
		$query->addWhere('r.role = ?', array(\Role::WR));

		// bereits eingesetzte wr auslassen
		$query->addWhere('(SELECT COUNT(as1.id) FROM Assignment as1 WHERE as1.team_id = ? AND as1.user_id = u.id) = 0', array($team->id));

		// nur wr mit freigabe abfragen
		if ($freigabeBeruecksichtigen) {
			$query->leftJoin('u.Availabilities av');
			$query->addWhere('av.datum = ?', array($date));
		}

		// nur wr mit erforderlicher lizenz abfragen
		if ($lizenzErzwingen) {
			$query->addWhere('lkombi.prio <= ?', array($team->getHighestLicensePriorityRequired(\Dancetype::KOMBI)));
			$query->addWhere('llat.prio <= ?', array($team->getHighestLicensePriorityRequired(\Dancetype::LATEIN)));
			$query->addWhere('lst.prio <= ?', array($team->getHighestLicensePriorityRequired(\Dancetype::STANDARD)));
		}

		// nur verfuegbare wr abfragen
		if ($verfuegbarkeitBeruecksichtigen) {
			$query->addWhere('(SELECT COUNT(as2.id) FROM Assignment as2 INNER JOIN as2.WRTeam ws2 WHERE as2.state in (0, 1, 2) AND ws2.date_from = ? AND ws2.id <> ? AND as2.user_id = u.id) = 0', array($date, $team->id));
		}

		// herkunft filtern
		if ($herkunft != 'all') {
			if ($herkunft == 'int') {
				$query->addWhere('cl.ltv_id = ?', array($ownerLtv));
			} elseif ($herkunft == 'ext') {
				$query->addWhere('cl.ltv_id != ?', array($ownerLtv));
			} else {
				if (is_array($herkunft)) {
					$query->andWhereIn('cl.ltv_id', $herkunft);
				} else {
					$query->addWhere('cl.ltv_id = ?', array($herkunft));
				}
			}
		}

		// doppelte clubs vermeinden
		if ($doppelteClubsVermeiden) {
			$query->addWhere('u.club_id NOT IN (SELECT DISTINCT(us3.club_id) FROM Assignment as3 INNER JOIN as3.User us3 WHERE as3.state in (0, 1, 2) AND as3.team_id = ?)', array($team->id));
		}

		// mandanten einschränken
		if ($mandanten != null) {
			$mandantenIds = MandatorUtils::asIdArray($mandanten);
			if (count_safe($mandantenIds) > 0) {
				$query->andWhereIn('owner.id', $mandantenIds);
			}
		}

		$query->orderBy('u.surname ASC, u.name ASC');
		$query->groupBy('u.id');
		$query->distinct(true);

		return $query->execute();
	}

	/**
	 * @codeCoverageIgnore
	 */
	public function findCandidates_Old(\WRTeam $team, $lizenz = true, $verfuegbar = true, $herkunft = 'all', $doppelvermeiden = true, $freigabe = true, $owners = null)
	{
		$owner = $team->Event->Owner;
		$owner_ltv = \ZWE\Settings\SettingsRegistry::getInstance()->get('zwe_ltv')->getMandatorValue($owner->id);

		$sql = "
			SELECT {u.*}, {cl.*}, {lst.*}, {llat.*}, {lkombi.*}, {av.*}
			FROM
				users u
                                INNER JOIN system_grants sg ON u.id = sg.user_id
                                INNER JOIN roles r ON sg.role_id = r.id
				INNER JOIN clubs cl ON u.club_id = cl.id
                                INNER JOIN mandators owner ON cl.mandator_id = owner.id
                                INNER JOIN licenses lst ON u.license_st_effective_id = lst.id
                                INNER JOIN licenses llat ON u.license_lat_effective_id = llat.id
                                INNER JOIN licenses lkombi ON u.license_kombi_effective_id = lkombi.id
                                LEFT JOIN availabilities av ON u.id = av.user_id
			WHERE ";

		// nur aktivierte abfragen
		$sql .= "u.active = " . \User::STATE_ACTIVE;

		// nur wr-benutzer abfragen
		$sql .= " AND r.role = '" . \Role::WR . "'";

		// bereits eingesetzte wr auslassen
		$sql .= " AND 0 = (SELECT COUNT(as1.id) FROM assignments as1 WHERE as1.team_id = {$team->id} AND as1.user_id = u.id) ";

		// nur wr mit freigabe abfragen
		$sql .= " AND (av.datum = '" . $team->Event->getDateTimeObject('datum')->format('Y-m-d') . "' ";
		if (!$freigabe) {
			$sql .= " OR 0 = (SELECT COUNT(avs1.id) FROM availabilities avs1 WHERE avs1.user_id = u.id AND avs1.datum = '" . $team->Event->getDateTimeObject('datum')->format('Y-m-d') . "') ";
		}
		$sql .= ") ";

		// richtige lizenz erzwingen
		if ($lizenz) {
			$sql .= " AND lst.prio <= " . $team->getHighestLicensePriorityRequired(\Dancetype::STANDARD);
			$sql .= " AND llat.prio <= " . $team->getHighestLicensePriorityRequired(\Dancetype::LATEIN);
			$sql .= " AND lkombi.prio <= " . $team->getHighestLicensePriorityRequired(\Dancetype::KOMBI);
		}

		// verfuegbarkeit berücksichtigen
		if ($verfuegbar) {
			$sql .= " AND 0 = (SELECT COUNT(as2.id) FROM assignments as2 INNER JOIN wrteams ws2 ON as2.team_id = ws2.id INNER JOIN events es2 ON ws2.event_id = es2.id WHERE as2.state in (0, 1, 2) AND ((es2.datum = '" . $team->Event->getDateTimeObject('datum')->format('Y-m-d') . "' AND es2.id <> " . $team->Event->id . ") OR es2.id = " . $team->Event->id . ") AND as2.user_id = u.id) ";
		}

		// herkunft filtern
		if ($herkunft == 'all') {

		} elseif ($herkunft == 'int' || $herkunft == 'ext') {
			if ($herkunft == 'int') {
				$sql .= " AND cl.ltv_id = '{$owner_ltv}' ";
			} else {
				$sql .= " AND cl.ltv_id != '{$owner_ltv}' ";
			}
		} else {
			$sql .= " AND cl.ltv_id = '${herkunft}' ";
		}

		// doppelte clubs im team vermeiden
		if ($doppelvermeiden) {
			$sql .= " AND u.club_id NOT IN (SELECT DISTINCT(us3.club_id) FROM assignments as3 INNER JOIN users us3 ON as3.user_id = us3.id WHERE as3.state in (0,1,2) AND as3.team_id = {$team->id})";
		}

		// mandanten einschränken
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			$sql .= " AND owner.id IN (" . implode(',', $owner_ids) . ") ";
		}

		// gruppieren und sortieren
		$sql .= " ORDER BY u.surname, u.name";

		// discard E_NOTICE since there are undefined variables in Doctrine_RawSql
		$err_rep = error_reporting();
		error_reporting(E_ALL ^ E_NOTICE);

		$query = new \Doctrine_RawSql();
		$query->parseDqlQuery($sql);
		$query->addComponent('u', 'User');
		$query->addComponent('sg', 'User.SystemGrants sg');
		$query->addComponent('r', 'SystemGrant.Role r');
		$query->addComponent('cl', 'User.Club cl');
		$query->addComponent('lst', 'User.LicenseStEffective lst');
		$query->addComponent('llat', 'User.LicenseLatEffective llat');
		$query->addComponent('lkombi', 'User.LicenseKombiEffective lkombi');
		$query->addComponent('av', 'User.Availabilities av');
		$query->addComponent('owner', 'Club.Owner owner');

		// reset error reporting
		error_reporting($err_rep);
		return $query->execute();
	}

}
