<?php
namespace ZWE\Repository;

use function ZWE\count_safe;

class UserRepository extends AbstractRepository
{

	/**
	 * @param null $params
	 * @return \Doctrine_Collection
	 */
	public function dynamicQuery($params = NULL)
	{
		$query = \Doctrine_Query::create()
			->select('u.*, c.*')
			->from('User u')
			->innerJoin('u.Club c')
			->orderBy('u.surname, u.name');
		if ($params != NULL) {
			if (!is_array($params)) {
				$query_params = array($params);
			} else {
				$query_params = $params;
			}
			$this->augmentDynamicQuery($query, $query_params);
		}
		return $query->execute();
	}

	/**
	 * @param $id
	 * @return \User
	 */
	public function find($id)
	{
		return $this->findInternal('User', $id);
	}

	/**
	 * @return mixed
	 */
	public function findAll()
	{
		return \Doctrine::getTable('User')->findAll();
	}

	public function findByActivationCode($activationcode)
	{
		$query = \Doctrine_Query::create()
			->from('User u')
			->where('u.activation_code = ?', $activationcode);
		return $query->fetchOne();
	}

	public function findByIdAndState($id, $state)
	{
		$user = $this->load($id);
		if ($user->active != $state) {
			$msg = 'Benutzerkonto hat falschen Status.';
			switch ($user->active) {
				case \User::STATE_INACTIVE:
					$msg = 'Benutzerkonto hat Status \'deaktiviert\'.';
					break;
				case \User::STATE_ACTIVE:
					$msg = 'Benutzerkonto hat Status \'aktiviert\'.';
					break;
				case \User::STATE_VALIDATED:
					$msg = 'Benutzerkonto hat Status \'validiert\'.';
					break;
				case \User::STATE_REGISTERED:
					$msg = 'Benutzerkonto hat Status \'registriert\'.';
					break;
			}
			throw new ObjectNotFoundException($msg);
		}
		return $user;
	}

	public function findByIdAndPasswordchangeCode($id, $passwordchangecode)
	{
		$query = \Doctrine_Query::create()
			->from('User u')
			->where('u.id = ?', $id)
			->andWhere('u.chpw_code = ?', $passwordchangecode);
		return $query->fetchOne();
	}

	public function findByLicense(\License $st, \License $lat, \License $kombi, $origin = 'all', $ltv_id = '', $owners = null, $exact = true)
	{
		$query = \Doctrine_Query::create()
			->from('User u')
			->innerJoin('u.LicenseStEffective lic_st_effective')
			->innerJoin('u.LicenseLatEffective lic_lat_effective')
			->innerJoin('u.LicenseKombiEffective lic_kombi_effective')
			->innerJoin('u.Club c')
			->orderBy('u.surname, u.name');
		if ($exact) {
			if ($st != null && $st->prio != 255) {
				$query->andWhere('lic_st_effective.id = ?', $st->id);
			}
			if ($lat != null && $lat->prio != 255) {
				$query->andWhere('lic_lat_effective.id = ?', $lat->id);
			}
			if ($kombi != null && $kombi->prio != 255) {
				$query->andWhere('lic_kombi_effective.id = ?', $kombi->id);
			}
		} else {
			if ($st != null) {
				$query->andWhere('lic_st_effective.prio <= ?', $st->prio);
			}
			if ($lat != null) {
				$query->andWhere('lic_lat_effective.prio <= ?', $lat->prio);
			}
			if ($kombi != null) {
				$query->andWhere('lic_kombi_effective.prio <= ?', $kombi->prio);
			}
		}
		if ($origin == 'int' || $origin == 'ext') {
			if ($ltv_id != '') {
				$op = $origin == 'ext' ? '!=' : '=';
				$query->andWhere('c.ltv_id ' . $op . ' ?', $ltv_id);
			}
		}
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			if (count_safe($owner_ids) > 0) {
				$query->andWhereIn('c.mandator_id', $owner_ids);
			}
		}
		$query->andWhere('u.active = ?', \User::STATE_ACTIVE);
		return $query->execute();
	}

	/**
	 * @param $username
	 * @return \User|null
	 */
	public function findByLogin($username)
	{
		if (trim($username) == '') {
			return null;
		}
		$query = \Doctrine_Query::create()
			->from('User u')
			->where('u.dtvnr = ?', $username);
		return $query->fetchOne();
	}

	/**
	 * @param $dtvNummer
	 * @return \User|null
	 */
	public function findByDTVNummer($dtvNummer)
	{
		if ($dtvNummer) {
			return \Doctrine_Query::create()
				->from('User u')
				->where('u.dtvnr = ?', $dtvNummer)
				->fetchOne();
		}
	}

	/**
	 * @param $username
	 * @param $email
	 * @return \User|null
	 */
	public function findByLoginAndEMail($username, $email)
	{
		if (trim($username) == '') {
			return null;
		}
		$query = \Doctrine_Query::create()
			->from('User u')
			->where('dtvnr = ?', $username)
			->andWhere('email = ?', $email);
		return $query->fetchOne();
	}

	public function findWR($state = \User::STATE_ACTIVE)
	{
		$query_params = array();
		if ($state !== null) {
			$query_params[] = QueryParam::where('u.active', '=', $state);
		}
		return $this->dynamicQuery($query_params);
	}

	public function findWRByStateAndOrigin($state = null, $origin = 'all', $ltv_id = '', $owners = null, $lastLicenseYear = null, $lastLicenseYearAndEarlier = false)
	{
		$query = \Doctrine_Query::create()
			->from('User u')
			->innerJoin('u.Roles r')
			->leftJoin('u.Club c')
			->leftJoin('c.LTV l')
			->where('r.role = ?', \Role::WR)
			->orderBy('u.surname, u.name');
		if ($state != null && is_array($state) && count_safe($state) > 0) {
			$query->andWhereIn('u.active', $state);
		}
		if ($origin == 'int' || $origin == 'ext') {
			if ($ltv_id != '') {
				$op = $origin == 'ext' ? '!=' : '=';
				$query->andWhere('c.ltv_id ' . $op . ' ?', $ltv_id);
			}
		}
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			if (count_safe($owner_ids) > 0) {
				$query->andWhereIn('c.mandator_id', $owner_ids);
			}
		}
		if ($lastLicenseYear != null) {
			if ($lastLicenseYearAndEarlier) {
				$query->andWhere('IFNULL(u.last_license_year, 1970) <= ?', $lastLicenseYear);
				$query->andWhere("l.id <> 'EXT'");
			} else {
				$query->andWhere('u.last_license_year = ?', $lastLicenseYear);
			}
		}
		return $query->execute();
	}

	/**
	 * @param $page
	 * @param $results
	 * @param \Mandator $mandator
	 * @return \Doctrine_Pager
	 */
	public function getLatestEventsPager($page, $results, \Mandator $mandator = null)
	{
		return new \Doctrine_Pager($this->getLatestEventsQuery($mandator), $page, $results);
	}

	/**
	 * @param \Mandator $mandator
	 * @return \Doctrine_Query
	 */
	public function getLatestEventsQuery(\Mandator $mandator = null)
	{
		$query = \Doctrine_Query::create()
			->select('u.*, IFNULL(u.activated, IFNULL(u.validated, IFNULL(u.registered, 0))) AS event_timestamp')
			->from('User u')
			->innerJoin('u.SystemGrants sg')
			->innerJoin('u.Club c')
			->innerJoin('sg.Role r')
			->where('r.role = ?', \Role::WR)
			->having('event_timestamp > ?', 0)
			->orderBy('event_timestamp DESC, u.surname, u.name');
		if ($mandator != null) {
			$query->andWhere('c.mandator_id = ?', $mandator->id);
		}
		return $query;
	}

	/**
	 * @param int $page
	 * @param int $maxPerPage
	 * @param bool $wr
	 * @param string $sortname
	 * @param string $sortorder
	 * @param string $queryfield
	 * @param string $queryvalue
	 * @param null $owners
	 * @return \Doctrine_Collection
	 */
	public function pagedQuery($page = 1, $maxPerPage = 10, $wr = true, $sortname = 'surname', $sortorder = 'asc', $queryfield = 'surname', $queryvalue = '%', $owners = null)
	{
		return $this->pagedQueryWithPagerResult($page, $maxPerPage, $wr, $sortname, $sortorder, $queryfield, $queryvalue, $owners)->execute();
	}

	/**
	 * @param int $page
	 * @param int $maxPerPage
	 * @param bool $wr
	 * @param string $sortname
	 * @param string $sortorder
	 * @param string $queryfield
	 * @param string $queryvalue
	 * @param null $owners
	 * @param null $status
	 * @return \Doctrine_Pager
	 */
	public function pagedQueryWithPagerResult($page = 1, $maxPerPage = 10, $wr = true, $sortname = 'surname', $sortorder = 'asc', $queryfield = 'surname', $queryvalue = '%', $owners = null, $status = null)
	{
		$query = \Doctrine_Query::create()
			->from('User u')
			->innerJoin('u.Roles r')
			->leftJoin('u.Club c')
			->where("u.{$queryfield} LIKE ?", $queryvalue)
			->orderBy("u.{$sortname} {$sortorder}");
		if ($wr) {
			$query->andWhere('r.role = ?', 'ROLE_WR');
		} else {
			$query->andWhere('r.role != ?', 'ROLE_WR');
		}
		if ($owners != null) {
			if (!$wr) {
				throw new \InvalidArgumentException("Die Abfrage nach ZWE-Benutzers darf nicht auf Mandanten eingeschränkt sein!");
			}
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			$query->andWhereIn('c.mandator_id', $owner_ids);
		}
		if ($status != null) {
			$query->andWhere('u.active = ?', $status);
		}
		return new \Doctrine_Pager($query, $page, $maxPerPage);
	}

}
