<?php

namespace ZWE\Queue;

use Carbon\Carbon;
use Psr\Log\LoggerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use ZWE\Container\ZWEApplicationContainer;

class QueueWorker
{

	/**
	 * @var ZWEApplicationContainer
	 */
	private $app;

	/**
	 * @var string
	 */
	private $pidFile;

	/**
	 * @var string
	 */
	private $statisticsFile;

	/**
	 * @var string
	 */
	private $stateFile;

	/**
	 * @var LoggerInterface
	 */
	private $logger;

	/**
	 * @var ValidatorInterface
	 */
	private $validator;

	/**
	 * @var QueueWorkerStatistics
	 */
	private $workerStatistics;

	/**
	 * @var DatabaseQueueDriver
	 */
	private $driver;

	private $shouldQuit = false;
	private $paused = false;
	private $options;

	public function __construct(ZWEApplicationContainer $app, $workingDirectory)
	{
		if (!$workingDirectory) {
			throw new \InvalidArgumentException('Working directory required!');
		}
		if (!file_exists($workingDirectory)) {
			if (!mkdir($workingDirectory)) {
				throw new \RuntimeException('Could not create working directory!');
			}
		} else {
			if (!is_dir($workingDirectory)) {
				throw new \InvalidArgumentException("{$workingDirectory} exists but is not a directory!");
			}
		}

		// if pcntl not loaded (e.g. in web env), we need these constants for posix_kill
		$defines = [
			'SIGKILL' => 9,
			'SIGUSR2' => 12,
			'SIGTERM' => 15,
			'SIGCONT' => 18,
		];
		foreach ($defines as $var => $value) {
			if (!defined($var)) {
				define($var, $value);
			}
		}

		$this->app = $app;
		$this->pidFile = $workingDirectory . DIRECTORY_SEPARATOR . 'worker.pid';
		$this->statisticsFile = $workingDirectory . DIRECTORY_SEPARATOR . 'worker.statistics';
		$this->stateFile = $workingDirectory . DIRECTORY_SEPARATOR . 'worker.state';
		$this->logger = $app->getLoggingService()->createLogger(QueueWorker::class);
		$this->validator = $app->getValidator();

		$this->options = (object)[
			'sleep' => 10,  // seconds
			'memory' => 56, // mb
			'lifetime' => 24 * 60 * 60 // seconds
		];
		$this->driver = new DatabaseQueueDriver($app->getValidator(), $app->getLoggingService());
	}

	// -- management functions for the queue worker daemon --- //

	/**
	 * Checks if the queue worker daemon is running
	 *
	 * @return bool
	 * @throws \Exception if the pid file could not be opened
	 */
	public function isRunning()
	{
		if (!file_exists($this->pidFile)) {
			return false;
		}

		if (($lock = @fopen($this->pidFile, 'c+')) === false) {
			throw new \Exception('Unable to open pid file ' . $this->pidFile);
		}
		if (flock($lock, LOCK_EX | LOCK_NB)) {
			return false;
		} else {
			flock($lock, LOCK_UN);
			return true;
		}
	}

	/**
	 * Returns statistics for the queue worker
	 *
	 * @return null|QueueWorkerStatistics
	 * @throws \Exception if the queue worker daemon is not running
	 */
	public function getWorkerStatistics()
	{
		if (!$this->isRunning()) {
			return;
		}

		if (file_exists($this->statisticsFile)) {
			return unserialize(file_get_contents($this->statisticsFile));
		}
	}

	/**
	 * Tries to stop the queue worker daemon by signalling SIGTERM
	 *
	 * @throws \Exception if the queue worker daemon is not running
	 */
	public function stopDaemon()
	{
		$pid = $this->getDaemonPid();
		if ($pid) {
			if (!posix_kill($pid, SIGTERM)) {
				throw new \RuntimeException('Could not send signal SIGTERM to process #' . $pid);
			}
		}
	}

	/**
	 * Tries to kill the queue worker daemon by signalling SIGKILL
	 *
	 * @throws \Exception if the queue worker daemon is not running
	 */
	public function killDaemon()
	{
		$pid = $this->getDaemonPid();
		if ($pid) {
			if (!posix_kill($pid, SIGKILL)) {
				throw new \RuntimeException('Could not send signal SIGKILL to process #' . $pid);
			}
		}
	}

	/**
	 * Tries to pause the queue worker daemon by signalling SIGUSR2
	 *
	 * @throws \Exception if the queue worker daemon is not running
	 */
	public function pauseDaemon()
	{
		$pid = $this->getDaemonPid();
		if ($pid) {
			if (!posix_kill($pid, SIGUSR2)) {
				throw new \RuntimeException('Could not send signal SIGUSR2 to process #' . $pid);
			}
		}
	}

	/**
	 * Tries to unpause the queue worker daemon by signalling SIGCONT
	 *
	 * @throws \Exception if the queue worker daemon is not running
	 */
	public function unpauseDaemon()
	{
		$pid = $this->getDaemonPid();
		if ($pid) {
			if (!posix_kill($pid, SIGCONT)) {
				throw new \RuntimeException('Could not send signnal SIGCONT to process #' . $pid);
			}
		}
	}

	/**
	 * Returns the pid of the queue worker daemon
	 *
	 * @return int
	 * @throws \Exception if the queue worker daemon is not running or the pid file could not be read
	 */
	private function getDaemonPid()
	{
		if (!$this->isRunning()) {
			throw new \RuntimeException('Queue worker is not running!');
		}

		$pid = @fopen($this->pidFile, 'r');
		if ($pid) {
			$value = fread($pid, 8192);
			fclose($pid);
			if ($value > 1) {
				return intval($value);
			} else {
				throw new \RuntimeException('Illegal process ID!');
			}
		} else {
			throw new \RuntimeException('Could not read pid file ' . $this->pidFile);
		}
	}

	// -- functions for the daemon itself -- //

	/**
	 * Starts the queue worker daemon
	 *
	 * @throws \Exception
	 */
	public function daemon()
	{
		if ($this->isRunning()) {
			throw new \RuntimeException('Queue worker daemon is already running');
		}

		if (($lock = @fopen($this->pidFile, 'c+')) === false) {
			throw new \Exception('Unable to open pid file ' . $this->pidFile);
		}
		if (!flock($lock, LOCK_EX | LOCK_NB)) {
			throw new \Exception('Could not acquire lock for ' . $this->pidFile);
		}

		$pid = getmypid();

		fseek($lock, 0);
		ftruncate($lock, 0);
		fwrite($lock, $pid);
		fflush($lock);

		if (file_exists($this->stateFile)) {
			$this->paused = trim(file_get_contents($this->stateFile)) == QueueWorkerStatistics::STATE_PAUSED;
			if ($this->paused) {
				$this->logger->info('last worker was paused before being stopped or killed, restoring state');
			} else {
				$this->logger->debug('last worker was running before being stopped or killed, continue with normal operations');
			}
		}

		$this->workerStatistics = new QueueWorkerStatistics($pid, $this->getMemoryUsage(), $this->paused ? QueueWorkerStatistics::STATE_PAUSED : QueueWorkerStatistics::STATE_IDLE);

		declare(ticks=1);

		register_shutdown_function(function () {
			try {
				\Doctrine_Query::create()->update(\QueuedJob::class)
					->set('status', \QueuedJob::JOB_ERROR)
					->where('status = ?', \QueuedJob::JOB_RUNNING)
					->execute();
			} catch (\Exception $e) {
				// swallow
			}
		});

		$this->logger->info('starting queue worker daemon', ['pid' => getmypid()]);
		if ($this->supportsAsyncSignals()) {
			$this->logger->debug('async signals supported, setting up listeners', ['pid' => getmypid()]);
			$this->listenForSignals();
		}
		while (true) {
			if (!$this->daemonShouldRun()) {
				$this->updateStatistics(QueueWorkerStatistics::STATE_PAUSED, null);
				$this->pauseWorker($this->options, null);
				continue;
			}
			$this->updateStatistics(QueueWorkerStatistics::STATE_IDLE, null);
			$this->runNextJob();
			$this->sleep($this->options->sleep);
			$this->stopIfNecessary($this->options, null);
		}
	}

	protected function runNextJob()
	{
		try {
			$queuedJob = $this->driver->getNextJob();
			if ($queuedJob) {
				$conn = \Doctrine_Manager::getInstance()->getCurrentConnection();

				// cancel jobs with too many retries
				if ($queuedJob->tries >= 3) {
					$this->logger->warning("queued job #{$queuedJob->id} exceeds retry limit, marking it failed");
					try {
						$conn->beginTransaction();
						$queuedJob->status = \QueuedJob::JOB_FAILED;
						$queuedJob->setDateTimeObject('finished', new \DateTime());
						$queuedJob->save();
						$conn->commit();
					} catch (\Exception $e) {
						$conn->rollback();
					}
					return;
				}

				$this->logger->info("running queued job #{$queuedJob->id}");
				try {
					$job = unserialize($queuedJob->payload);

					// cancel invalid jobs
					$violations = $this->validator->validate($job);
					if ($violations->count() > 0) {
						$this->logger->error('job is invalid', ['violations' => $violations]);
						try {
							$conn->beginTransaction();
							$queuedJob->status = \QueuedJob::JOB_FAILED;
							$queuedJob->setDateTimeObject('finished', new \DateTime());
							$queuedJob->log = serialize([
								'violations' => $violations
							]);
							$queuedJob->save();
							$conn->commit();
						} catch (\Exception $e) {
							$conn->rollback();
						}
						return;
					}

					// update the queued job
					$conn->beginTransaction();
					$queuedJob->status = \QueuedJob::JOB_RUNNING;
					$queuedJob->tries++;
					$queuedJob->setDateTimeObject('started', new \DateTime());
					$queuedJob->save();
					$conn->commit();

					// update the state of the queue worker
					$this->updateStatistics(QueueWorkerStatistics::STATE_WORKING, $queuedJob->id);

					// run the job
					$completed = $job->run(function () {
						return $this->paused || $this->shouldQuit;
					});

					// update the queued job
					$conn->beginTransaction();
					if ($completed) {
						$this->logger->info("job #{$queuedJob->id} was completed fully, marking it finished");
						$queuedJob->status = \QueuedJob::JOB_FINISHED;
						$queuedJob->setDateTimeObject('finished', new \DateTime());
					} else {
						$this->logger->info("job #{$queuedJob->id} was not completed fully and will be resumed");
						$queuedJob->status = \QueuedJob::JOB_QUEUED;
					}
					$queuedJob->save();
					$conn->commit();

					$this->workerStatistics->jobsProcessed++;
				} catch (\Exception $e) {
					$this->logger->error('exception during job execution', ['e' => $e]);
					try {
						$conn->rollback();
					} catch (\Exception $innerE1) {
						// cannot do anything here
					}

					try {
						$conn->beginTransaction();
						$queuedJob->status = \QueuedJob::JOB_ERROR;
						$queuedJob->log = serialize([
							'exception' => get_class($e),
							'message' => $e->getMessage(),
							'trace' => $e->getTraceAsString(),
						]);
						$queuedJob->save();
						$conn->commit();
					} catch (\Exception $innerE2) {
						$this->logger->error('exception during job update after error', ['e' => $innerE2]);
						$conn->rollback();
					}
				}
			}
		} catch (\Exception $e) {
			$this->logger->error('error loading next job', ['e' => $e]);
		}
	}

	private function daemonShouldRun()
	{
		return !$this->paused;
	}

	private function listenForSignals()
	{
		if (version_compare(PHP_VERSION, '7.1', '>=')) {
			pcntl_async_signals(true);
		}
		pcntl_signal(SIGTERM, function () {
			$this->logger->info('received SIGTERM signal, stopping worker', ['pid' => getmypid()]);
			$this->shouldQuit = true;
		});
		pcntl_signal(SIGUSR2, function () {
			if (!$this->paused) {
				$this->logger->info('received SIGUSR2 signal, pausing worker', ['pid' => getmypid()]);
				$this->updateStateFile(QueueWorkerStatistics::STATE_PAUSED);
				$this->paused = true;
			}
		});
		pcntl_signal(SIGCONT, function () {
			if ($this->paused) {
				$this->logger->info('received SIGCONT signal, unpausing worker', ['pid' => getmypid()]);
				$this->updateStateFile(QueueWorkerStatistics::STATE_IDLE);
				$this->paused = false;
			}
		});
	}

	private function supportsAsyncSignals()
	{
		return extension_loaded('pcntl');
	}

	private function kill($status = 0)
	{
		$this->logger->info('killing queue worker with SIGKILL', ['pid' => getmypid()]);
		posix_kill(getmypid(), SIGKILL);
		exit($status);
	}

	private function stop($status = 0)
	{
		$this->logger->info('stopping queue worker', ['pid' => getmypid()]);
		exit($status);
	}

	private function sleep($seconds)
	{
		if ($seconds < 1) {
			usleep($seconds * 1000000);
		} else {
			sleep($seconds);
		}
	}

	private function stopIfNecessary($options, $lastRestart)
	{
		if ($this->shouldQuit) {
			$this->kill();
		}
		if ($this->memoryExceeded($options->memory)) {
			$this->logger->info("memory usage exceeds configured limit of {$this->options->memory}M, quitting worker", ['pid' => getmypid()]);
			$this->stop(12);
		} elseif ($this->queueShouldRestart($options->lifetime)) {
			$this->logger->info("worker lifetime exceeds configured limit of {$this->options->lifetime}s, quitting worker", ['pid' => getmypid()]);
			$this->stop(12);
		}
	}

	private function memoryExceeded($memoryLimit)
	{
		return $this->getMemoryUsage() >= $memoryLimit;
	}

	private function getMemoryUsage()
	{
		return memory_get_usage() / 1024 / 1024;
	}

	private function queueShouldRestart($maxLifeTime)
	{
		$shouldQuitAt = $this->workerStatistics->startTime->copy()->addSecond($maxLifeTime);
		return $shouldQuitAt->isPast();
	}

	private function pauseWorker($options, $lastRestart)
	{
		$this->sleep($options->sleep > 0 ? $options->sleep : 1);
		$this->stopIfNecessary($options, $lastRestart);
	}

	private function updateStatistics($runState, $currentJob)
	{
		$this->workerStatistics->state = $runState;
		$this->workerStatistics->currentJob = $currentJob;
		$this->workerStatistics->memoryUsage = $this->getMemoryUsage();

		$state = @fopen($this->statisticsFile, 'w');
		if ($state) {
			fwrite($state, serialize($this->workerStatistics));
			fclose($state);
		}
	}

	private function updateStateFile($runState)
	{
		if (!file_put_contents($this->stateFile, $runState)) {
			$this->logger->warning('could not update state file with run state ' . $runState);
		}
	}
}
