<?php
namespace ZWE\Mandator;

use ZWE\Context\StaticContextHolder;
use ZWE\Singleton;

/**
 * Mandanten-Kontext, der den Zugriff auf den aktuellen Mandanten ermöglicht
 * (sofern vorhanden).
 *
 * Abhängig von der PHP-SAPI wird der Mandant entweder aus der HTTP-Session
 * oder aus einer statischen Variable geladen. Dafür stehen zwei
 * unterschiedliche MandatorContextHolder zur Verfügung.
 *
 * Bei Objekten dieser Klasse handelt es sich um Singletons, die nicht
 * geklont werden können.
 *
 * @see         MandatorContextHolder
 *
 * @author      Hendrik Heneke <hendrik@pixelwunder.net>
 * @since       2.5
 */
class MandatorContext extends Singleton
{

	private static $instance;

	private $contextHolder;

	private function  __construct()
	{
		if (strtolower(php_sapi_name()) == 'cli') {
			$this->contextHolder = new StaticContextHolder();
		} else {
			$this->contextHolder = new MandatorCISessionContextHolder();
		}
	}

	public function clear()
	{
		$this->contextHolder->clearContext();
	}

	/**
	 * Gibt den aktuellen Mandanten-Kontext zurück. Der Kontext wird
	 * erzeugt, falls noch keine Instanz existiert.
	 *
	 * @return MandatorContext Mandanten-Kontext
	 */
	public static final function getInstance()
	{
		if (self::$instance == null) {
			self::$instance = new MandatorContext();
		}
		return self::$instance;
	}


	/**
	 * Gibt den aktuellen Mandanten zurück.
	 *
	 * @return \Mandator|null
	 */
	public final function getMandator()
	{
		return $this->contextHolder->getContext();
	}

	/**
	 * Speichert den Mandanten im aktuellen Kontext.
	 *
	 * @param Mandator $mandator Mandant
	 */
	public final function setMandator(\Mandator $mandator)
	{
		$this->contextHolder->setContext($mandator);
	}

}