<?php

namespace ZWE\Mail;

use Psr\Log\LoggerInterface;
use ZWE\Container\ZWEApplicationContainer;
use ZWE\FileStorage\File;
use ZWE\Settings\SettingsRegistry;
use ZWE\UUID;
use function ZWE\count_safe;

class MailingService
{

	/**
	 * @var ZWEApplicationContainer
	 */
	private $container;

	/**
	 * @var LoggerInterface
	 */
	private $logger;

	/**
	 * @var \Swift_Mailer
	 */
	private $mailer;

	/**
	 * @var \Swift_Signers_DKIMSigner|null
	 */
	private $signer;

	public function __construct(ZWEApplicationContainer $container)
	{
		$this->container = $container;
		$this->logger = $container->getLoggingService()->createLogger(MailingService::class);
		$this->mailer = $this->createMailer();
		$this->signer = $this->createSigner();
	}

	/**
	 * @return \Swift_Mailer
	 */
	private function createMailer()
	{
		$transport = null;
		$configurationSource = $this->container->getConfigurationSource();
		if (isset($configurationSource["protocol"])) {
			switch ($configurationSource["protocol"]) {
				case 'sendmail';
					$transport = new \Swift_SendmailTransport();
					break;
				case 'mail':
					$transport = new \Swift_MailTransport();
					break;
				case 'smtp':
					$transport = $this->createSmtpTransport();
					break;
				case 'none':
				default:
					$transport = new \Swift_NullTransport();
			}
		} else {
			$transport = new \Swift_NullTransport();
		}
		$mailer = new \Swift_Mailer($transport);

		if (strpos(php_sapi_name(), 'cli') !== false) {
			$maxMessagesPerMinute = SettingsRegistry::getInstance()->get('mail_max_messages_per_minute')->getValue();
			if ($maxMessagesPerMinute > 0) {
				$this->logger->debug("adding throttler plugin; max messages per minute = {$maxMessagesPerMinute}");
				$mailer->registerPlugin(new \Swift_Plugins_ThrottlerPlugin($maxMessagesPerMinute, \Swift_Plugins_ThrottlerPlugin::MESSAGES_PER_MINUTE));
			}
		}
		return $mailer;
	}

	/**
	 * @return \Swift_SmtpTransport
	 */
	private function createSmtpTransport()
	{
		$transport = new \Swift_SmtpTransport();
		$configurationSource = $this->container->getConfigurationSource();
		if (isset($configurationSource["smtp_host"])) {
			$transport->setHost($configurationSource["smtp_host"]);
		}
		if (isset($configurationSource["smtp_port"])) {
			$transport->setPort($configurationSource["smtp_port"]);
		}
		if (isset($configurationSource["smtp_user"])) {
			$transport->setUsername($configurationSource["smtp_user"]);
		}
		if (isset($configurationSource["smtp_pass"])) {
			$transport->setPassword($configurationSource["smtp_pass"]);
		}
		// ssl oder tls
		if (isset($configurationSource["smtp_security"])) {
			$transport->setEncryption($configurationSource["smtp_security"]);
		}
		// array('ssl' => array('allow_self_signed' => true, 'verify_peer' => false, 'verify_host' => false, 'verify_peer_name' => false))
		if (isset($configurationSource["smtp_transport_stream_options"])) {
			$transport->setStreamOptions($configurationSource["smtp_transport_stream_options"]);
		}
		return $transport;
	}

	/**
	 * @return \Swift_Signers_DKIMSigner|null
	 */
	private function createSigner()
	{
		return null;
	}

	/**
	 * Versendet die Passworterinnerung.
	 *
	 * @param string $messageId
	 * @param \User $user
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendPasswordResetMessage($messageId, \User $user, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createPasswordResetMessage($user);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('password reset message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('password reset message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Aktivierungsnachricht
	 *
	 * @param string $messageId
	 * @param \User $user
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendUserActivatedMessage($messageId, \User $user, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createUserActivatedMessage($user);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('user activated message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('user activated message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet eine Nachricht über einen geänderten Login
	 *
	 * @param string $messageId
	 * @param \User $user
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendUserLoginChangedMessage($messageId, \User $user, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createUserLoginChangedMessage($user);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('user login changed message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('user login changed message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Validierungsnachricht
	 *
	 * @param string $messageId
	 * @param \User $user
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendUserValidationMessage($messageId, \User $user, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createUserValidationMessage($user);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('user validation message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('user validation message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Freischaltungsnachricht
	 *
	 * @param string $messageId
	 * @param \User $user
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendUserUnlockedMessage($messageId, \User $user, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createUserUnlockedMessage($user);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('user unlocked message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('user unlocked message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Einladungsnachricht
	 *
	 * @param string $messageId
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendInvitationMessage($messageId, \Assignment $einsatz, $bemerkungen, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createInvitationMesssage($einsatz, $bemerkungen);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('invitation message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('invitation message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet manuell erfasste Absage
	 *
	 * @param string $messageId
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendAssignmentCancelledMessage($messageId, \Assignment $einsatz, $bemerkungen, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createAssignmentCancelledMessage($einsatz, $bemerkungen);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('assignment manually cancelled message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('assignment manually cancelled message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet eine manuell erfasste Zusage
	 *
	 * @param string $messageId
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendAssignmentConfirmedMessage($messageId, \Assignment $einsatz, $bemerkungen, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createAssignmentConfirmedMessage($einsatz, $bemerkungen);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('assignment manually confirmed message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('assignment manually confirmed message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet eine Einsatzlöschung
	 *
	 * @param string $messageId
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendAssignmentDeletedMessage($messageId, \Assignment $einsatz, $bemerkungen, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createAssignmentDeletedMessage($einsatz, $bemerkungen);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('assignment deleted message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('assignment deleted message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet Erinnerungen an Zu- oder Absage
	 *
	 * @param string $messageId
	 * @param \Assignment $einsatz
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendReminderMessage($messageId, \Assignment $einsatz, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createReminderMessage($einsatz);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('reminder message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('reminder message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Absage einer ganzen Veranstaltung
	 *
	 * @param string $messageId
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendEventCancelledMessage($messageId, \Assignment $einsatz, $bemerkungen, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createEventCancelledMessage($einsatz->WRTeam->Event, $einsatz->WRTeam, $einsatz, $bemerkungen);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('event cancelled message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('event cancelled message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Absage einer Wertungsrichters
	 *
	 * @param string $messageId
	 * @param \Assignment $einsatz
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendCancellationMessage($messageId, \Assignment $einsatz, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createCancellationMessage($einsatz);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('cancellation message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('cancellation message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Nachricht für die Validierung einer neuer E-Mailadresse
	 *
	 * @param string $messageId
	 * @param \ChangeEmailRequest $request
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendValidationNewEmailMessage($messageId, \ChangeEmailRequest $request, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createValidateNewEmailMessage($request);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('validate new email message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('cvalidate new email delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * @param string $messageId
	 * @param \Mandator $mandant
	 * @param string $messageId
	 * @param string $subject
	 * @param string $mailHtml
	 * @param string $mailText
	 * @param array $bcc
	 * @param null|string|string[] $attachments
	 * @param callable $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendCustomMessage($messageId, \Mandator $mandant, $subject, $mailHtml, $bcc = array(), $attachments = null, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createCustomMessage($mandant, $subject, $mailHtml, $bcc, $attachments);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('custom message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('custom message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * @param string $messageId
	 * @param \Event $event
	 * @param File $pdf
	 * @param File $zwex
	 * @param mixed $cc
	 * @param callable $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendAssignmentplanMessage($messageId, \Event $event, File $pdf, File $zwex, $cc = null, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createAssignmentplanMessage($event, $pdf, $zwex);
		if ($cc !== null) {
			if (is_array($cc)) {
				foreach ($cc as $c) {
					$message->addCc($c);
				}
			} else {
				$message->addCc($cc);
			}
		}
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('assignment plan message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('assignment plan message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Versendet die Löschbenachrichtigung
	 *
	 * @param string $messageId
	 * @param \User $user
	 * @params callable|null $shouldInterrupt
	 * @return \Swift_Mime_SimpleMessage|null - the message if all chunks needed have been sent, null otherwise
	 */
	public function sendUserDeletedMessage($messageId, \User $user, callable $shouldInterrupt = null)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$message = $this->container->getMessageCreator()->createUserDeletedMessage($user);
		if ($this->sendMessage($messageId, $message, $shouldInterrupt)) {
			$this->logger->debug('user deleted message was sent, returning message');
			return $message;
		} else {
			$this->logger->debug('user deleted message delivery was interrupted, returning null');
			return null;
		}
	}

	/**
	 * Sends a message, chunked in severals slices if needed.
	 *
	 * @param string $messageId
	 * @param \Swift_Mime_SimpleMessage $message
	 * @param callable|null $shouldInterrupt
	 * @return boolean returns true if all slices have been sent, false if otherwise
	 */
	private function sendMessage($messageId, \Swift_Mime_SimpleMessage $message, callable $shouldInterrupt = null)
	{
		$maxRecipients = $this->container->getSettingsService()->get('mail_max_recipients_per_message')->getValue();
		// check if this message is a chunked delivery that was interrupted
		$chunkedDelivery = $this->findChunkedDelivery($messageId);
		if ($chunkedDelivery) {
			$maxRecipients = $chunkedDelivery->max_recipients;
			$this->logger->debug("chunked delivery of message ${messageId} was interrupted; max_recipients={$chunkedDelivery->max_recipients}, chunks_sent={$chunkedDelivery->chunks_sent}");
		}

		// check if there is a recipient limit per message
		if ($maxRecipients > 0) {
			$this->logger->debug("checking if message exceeds configured limit of recipients; max recipients per message = {$maxRecipients}");
			$count = count_safe($message->getTo()) + count_safe($message->getCc()) + count_safe($message->getBcc());
			// number of recipients exceeds configured limit, have to send this in chunks
			if ($count > $maxRecipients) {
				if (!$chunkedDelivery) {
					$chunkedDelivery = new \ChunkedDelivery();
					$chunkedDelivery->id = $messageId;
					$chunkedDelivery->max_recipients = $maxRecipients;
					$chunkedDelivery->chunks_sent = 0;
				}

				$chunks = ceil($count / $maxRecipients);
				$this->logger->debug("message has {$count} recipients and needs to be split into ${chunks} chunks");
				$allRecipients = [];
				if ($message->getTo()) {
					foreach ($message->getTo() as $key => $value) {
						$allRecipients[] = (object)[
							'type' => 'to',
							'value' => is_numeric($key) ? $value : [$key => $value]
						];
					}
				}
				if ($message->getCc()) {
					foreach ($message->getCc() as $key => $value) {
						$allRecipients[] = (object)[
							'type' => 'cc',
							'value' => is_numeric($key) ? $value : [$key => $value]
						];
					}
				}
				if ($message->getBcc()) {
					foreach ($message->getBcc() as $key => $value) {
						$allRecipients[] = (object)[
							'type' => 'bcc',
							'value' => is_numeric($key) ? $value : [$key => $value]
						];
					}
				}

				$startAt = $chunkedDelivery->chunks_sent + 1;
				if ($startAt > 1) {
					$this->logger->debug("resuming chunked delivery with chunk {$startAt}");
				}
				for ($i = $startAt; $i <= $chunks; $i++) {
					$offset = ($i - 1) * $maxRecipients;
					$currentChunkOfRecipients = array_slice($allRecipients, $offset, $maxRecipients);

					$message->setTo([]);
					$message->setCc([]);
					$message->setBcc([]);

					foreach ($currentChunkOfRecipients as $r) {
						switch ($r->type) {
							case 'to':
								if (is_array($r->value)) {
									$address = array_keys($r->value)[0];
									$message->addTo($address, $r->value[$address]);
								} else {
									$message->addTo($r->value);
								}
								break;
							case 'cc':
								if (is_array($r->value)) {
									$address = array_keys($r->value)[0];
									$message->addCc($address, $r->value[$address]);
								} else {
									$message->addCc($r->value);
								}
								break;
							default:
								if (is_array($r->value)) {
									$address = array_keys($r->value)[0];
									$message->addBcc($address, $r->value[$address]);
								} else {
									$message->addBcc($r->value);
								}
						}
					}

					// check if job should be interrupted
					if ($shouldInterrupt != null) {
						$interrupt = call_user_func($shouldInterrupt);
						if ($interrupt) {
							$this->logger->warning('chunked delivery of message needs to be interrupted, returning false');
							return false;
						}
					}

					$this->logger->debug('sending message chunk ' . $i);
					$conn = \Doctrine_Manager::getInstance()->getCurrentConnection();
					try {
						if ($shouldInterrupt) {
							$conn->beginTransaction();
						}
						$this->sendInternal($message);
						$chunkedDelivery->chunks_sent++;
						if ($shouldInterrupt) {
							$chunkedDelivery->save();
							$this->logger->debug('saved state of chunked delivery', ['chunkedDelivery' => $chunkedDelivery->toArray()]);
							$conn->commit();
						}
					} catch (\Exception $e) {
						if ($shouldInterrupt) {
							$conn->rollback();
						}
						$this->logger->error("exception during chunked delivery: {$e->getMessage()}", ['e' => $e]);
						throw $e;
					}
				}

				if ($shouldInterrupt) {
					$conn = \Doctrine_Manager::getInstance()->getCurrentConnection();
					try {
						$conn->beginTransaction();
						$chunkedDelivery->delete();
						$conn->commit();
					} catch (\Exception $e) {
						$conn->rollback();
						$this->logger->warning("could not delete state of chunked delivery: {$e->getMessage()}", ['chunkedDelivery' => $chunkedDelivery->toArray(), 'e' => $e]);
					}
				}

				// all chunks have been sent, return true
				return true;
			}
		}

		$this->sendInternal($message);
		// only one chunk, return true
		return true;
	}

	private function sendInternal(\Swift_Mime_SimpleMessage $message)
	{
		$failedRecipients = [];
		if ($this->mailer->getTransport()->isStarted()) {
			$this->mailer->getTransport()->stop();
		}
		$message->setId(time() . '.' . UUID::randomUUID() . '@' . $this->findHostname());
		if ($this->signer) {
			$this->signer->reset();
			$message->attachSigner($this->signer);
		}
		$this->mailer->send($message, $failedRecipients);
		if (count_safe($failedRecipients) > 0) {
			$this->logger->warning('delivery of message ' . $message->getId() . ' failed for the following recipients: ' . implode(', ', $failedRecipients));
		}
	}

	private function findHostname()
	{
		if (isset($_SERVER['SERVER_NAME'])) {
			return $_SERVER['SERVER_NAME'];
		} else if (isset($_SERVER['HOSTNAME'])) {
			return $_SERVER['HOSTNAME'];
		} else if (($uname = php_uname('n'))) {
			return $uname;
		} else {
			return 'zwe';
		}
	}

	/**
	 * @param $messageId
	 * @return \ChunkedDelivery|null
	 */
	private function findChunkedDelivery($messageId)
	{
		if (!$messageId) {
			throw new \InvalidArgumentException('Message-ID required!');
		}

		$results = \Doctrine_Query::create()->from(\ChunkedDelivery::class)->where('id = ?', [$messageId])->limit(1)->execute();
		if ($results && count_safe($results) > 0) {
			return $results[0];
		}
	}
}
