<?php
namespace ZWE\Images;

use ZWE\FileStorage\File;
use function ZWE\count_safe;

class ImageValidator
{

	/**
	 * @var \ZWE\FileStorage\File
	 */
	private $file;

	/**
	 * @var array
	 */
	private $descriptor;

	/**
	 * @var array
	 */
	private $errors = array();

	public function __construct(File $file, array $descriptor)
	{
		$this->file = $file;
		$this->descriptor = $descriptor;
		$this->validateInternal();
	}

	private function validateInternal()
	{
		if (!$this->file->exists()) {
			$this->errors[] = "Datei existiert nicht.";
			return;
		}

		if (!isset($this->descriptor['mime'])) {
			$this->errors[] = "Ungültiger Deskriptor (kein MIME-Typ).";
			return;
		}

		$info = getimagesize($this->file->getAbsolutePath());
		if (!$info) {
			$this->errors[] = "Bilddatei ist ungültig";
			return;
		}

		if ($info['mime'] != $this->descriptor['mime']) {
			$this->errors[] = "Bilddatei ist nicht vom Typ " . $this->descriptor['mime'] . ".";
			return;
		}

		$width = $info[0];
		if (isset($this->descriptor['minwidth'])) {
			if ($width < $this->descriptor['minwidth']) {
				$this->errors[] = "Bild ist nicht breiter als " . $this->descriptor['minwidth'] . "px.";
				return;
			}
		}
		if (isset($this->descriptor['maxwidth'])) {
			if ($width > $this->descriptor['maxwidth']) {
				$this->errors[] = "Bild ist breiter als " . $this->descriptor['maxwidth'] . "px.";
				return;
			}
		}

		$height = $info[1];
		if (isset($this->descriptor['minheight'])) {
			if ($height < $this->descriptor['minheight']) {
				$this->errors[] = "Bild ist nicht höher als " . $this->descriptor['minheight'] . "px.";
				return;
			}
		}
		if (isset($this->descriptor['maxheight'])) {
			if ($height > $this->descriptor['maxheight']) {
				$this->errors[] = "Bild ist höher als " . $this->descriptor['maxheight'] . "px.";
				return;
			}
		}

		$ratio = $width / $height;
		if (isset($this->descriptor['minratio'])) {
			if ($ratio < $this->descriptor['minratio']) {
				$this->errors[] = "Seitenverhältnis ist kleiner als " . $this->descriptor['minratio'];
				return;
			}
		}
		if (isset($this->descriptor['maxratio'])) {
			if ($ratio > $this->descriptor['maxratio']) {
				$this->errors[] = "Seitenverhältnis ist größer als " . $this->descriptor['maxratio'];
				return;
			}
		}
	}

	/**
	 * @return bool
	 */
	public function isValid()
	{
		return count_safe($this->errors) == 0;
	}

	/**
	 * @return array
	 */
	public function getErrors()
	{
		return $this->errors;
	}
}