<?php
namespace ZWE\Export\Einsatzplan;

use ZWE\FileStorage\File;
use ZWE\FileStorage\FileNotFoundException;
use ZWE\FileStorage\FileStorage;
use ZWE\Settings\SettingsRegistry;
use function ZWE\removeSpecialChars;

class ExportEinsatzplanPDFImpl extends \FPDF
{

	/**
	 * @var FileStorage
	 */
	private $resourceStorage;

	/**
	 * @var \ZWE\Settings\SettingsRegistry
	 */
	private $settingsRegistry;

	/**
	 * @var \Event
	 */
	private $event;

	/**
	 * @var boolean
	 */
	private $translitAvailable;

	public function __construct(FileStorage $resourceStorage, SettingsRegistry $settingsRegistry)
	{
		parent::__construct();
		$this->SetMargins(20, 50, 16.3);
		$this->SetAutoPageBreak(true, 33);
		$this->resourceStorage = $resourceStorage;
		$this->settingsRegistry = $settingsRegistry;

		// try to detect if we are running in a docker image which is presuambly based on alpine that does not allow iconv with TRANSLIT
		$this->translitAvailable = strpos(strtolower(php_uname()), 'linuxkit') === false;
	}

	public function getOutputFilename()
	{
		return sprintf('%s-%s_Einsatzplan.pdf', $this->event->getDateTimeObject('datum')->format('Ymd'), $this->event->id);
	}

	public function generate(\Event $event, $bemerkungen = null)
	{
		$this->event = $event;

		$this->AddPage();

		$this->SetFont('Arial', 'UB', 10);
		$this->Cell(0, 8, $this->utf8ToLatin1('Wertungsrichtereinsatzplan für ' . $event->name . ' am ' . $event->formatDate()), 0, 2, 'L');

		$this->Ln(4);

		$this->SetFont('Arial', 'B', 8);
		$this->SetTextColor(255, 0, 0);
		$this->SetDrawColor(255, 0, 0);
		if (SettingsRegistry::getInstance()->get('zwe_gdpr_enabled')->getSystemValue()) {
			$this->Cell(0, 5, $this->utf8ToLatin1('Dieses Dokument enthält persönliche Daten der Wertungsrichter. Die Veröffentlichung ist aus Datenschutzgründen untersagt.'), 'LTR', 2, 'L');
			$this->Cell(0, 5, $this->utf8ToLatin1('Gemäß Datenschutzgrundverordnung (DSGVO) sind Sie verpflichtet, die im Einsatzplan und der ZWE-Exportdatei gespeicherten'), 'LR', 2, 'L');
			$this->Cell(0, 5, $this->utf8ToLatin1('Daten der Wertungsrichter nach Abschluss der Turnierveranstaltung vollständig zu löschen (auch aus Ihrem Posteingang),'), 'LR', 2, 'L');
			$this->Cell(0, 5, $this->utf8ToLatin1('da die Zustimmung zur Datenverarbeitung nur bis zum Ende des jeweiligen Einsatzes erteilt wurde.'), 'LRB', 2, 'L');
		} else {
			$this->Cell(0, 8, $this->utf8ToLatin1('Dieses Dokument enthält persönliche Daten der Wertungsrichter. Die Veröffentlichung ist aus Datenschutzgründen untersagt.'), 1, 2, 'L');
		}
		$this->SetTextColor(0, 0, 0);
		$this->SetDrawColor(0, 0, 0);

		$this->Ln(4);

		$club = $event->Club;

		$this->SetFont('Arial', 'U', 8);
		$this->Cell(25, 4, 'Ausrichter', 0, 0, 'L');
		$this->SetFont('Arial', '', 8);
		$this->Cell(0, 4, $this->utf8ToLatin1($club->name), 0, 1, 'L');

		$this->SetFont('Arial', 'U', 8);
		$this->Cell(25, 4, 'Ansprechpartner', 0, 0, 'L');
		$this->SetFont('Arial', '', 8);
		$this->Cell(0, 4, $this->utf8ToLatin1($club->correspondent), 0, 1, 'L');
		$this->SetFont('Arial', '', 6);
		$this->Cell(25, 3, '', 0, 0, 'L');
		$this->Cell(0, 3, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull('', 'Telefon: %s', $club->phone) . \ZWE\StringUtils::formatIfnull('', ' / ', $club->phone) . \ZWE\StringUtils::formatIfnull('', 'E-Mail: %s', $club->email)), 0, 1, 'L');

		$this->SetFont('Arial', 'U', 8);
		$this->Cell(25, 4, $this->utf8ToLatin1('Turnierstätte'), 0, 0, 'L');
		$this->SetFont('Arial', '', 8);
		$location = $event->Location;
		$this->Cell(0, 4, $this->utf8ToLatin1($location->name), 0, 1, 'L');
		if ($location->address) {
			$this->Cell(25, 3, '', 0, 0, 'L');
			$this->SetFont('Arial', '', 6);
			$this->Cell(0, 3, $this->utf8ToLatin1($location->address), 0, 1, 'L');
		}
		if ($location->strasse && $location->Country) {
			$this->Cell(25, 3, '', 0, 0, 'L');
			$this->SetFont('Arial', '', 6);
			$this->Cell(0, 3, $this->utf8ToLatin1("{$location->strasse}, {$location->plz} $location->ort, {$location->Country->name}"), 0, 1, 'L');
		}

		$this->SetFont('Arial', 'U', 8);
		$this->Cell(25, 4, 'Stand', 0, 0, 'L');
		$this->SetFont('Arial', '', 8);
		$this->Cell(0, 4, date('d.m.Y H:i') . ' Uhr', 0, 1, 'L');
		$this->Cell(25, 3, '', 0, 0, 'L');
		$this->SetFont('Arial', 'I', 6);
		$this->Cell(0, 3, $this->utf8ToLatin1('Dieser Einsatzplan ersetzt alle ggf. vorher versandten Einsatzpläne!'), 0, 1, 'L');

		$this->Ln(4);

		$alleBemerkungen = '';
		if (!\ZWE\StringUtils::isnullString($event->bemerkungen)) {
			$alleBemerkungen .= $event->bemerkungen;
		}
		if (!\ZWE\StringUtils::isnullString($bemerkungen)) {
			$alleBemerkungen .= ' ';
			$alleBemerkungen .= $bemerkungen;
		}
		if (!\ZWE\StringUtils::isnullString($alleBemerkungen)) {
			$this->SetFont('Arial', 'U', 8);
			$this->Cell(25, 4, 'Bemerkungen', 0, 0, 'L');
			$this->SetFont('Arial', '', 8);
			$this->Write(4, $this->utf8ToLatin1($alleBemerkungen));
			$this->Ln(8);
		}

		foreach ($event->WRTeam as $team) {
			$this->SetFont('Arial', 'B', 10);
			$this->Cell(0, 5, $this->utf8ToLatin1($team->name), 0, 2, 'L');
			$this->SetFont('Arial', '', 8);
			$done = false;
			foreach ($team->getCompetitions() as $turnier) {
				$this->SetFont('Arial', 'U', 8);
				if (!$done) {
					$print = 'Turniere';
					$done = true;
				} else {
					$print = '';
				}

				$this->Cell(25, 3, $print, 0, 0, 'L');
				$this->SetFont('Arial', '', 8);
				$this->Cell(40, 4, $this->utf8ToLatin1($turnier->formatDate(), 0, 0, 'L'));
				$this->Cell(10, 4, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull('(k.A.)', '%s', $turnier->uhrzeit)), 0, 0, 'L');
				$this->Cell(15, 4, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull(' ', '%s', $turnier->Startgroup->name)), 0, 0, 'L');
				$this->Cell(10, 4, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull(' ', '%s', $turnier->Startclass->name)), 0, 0, 'L');
				$this->Cell(10, 4, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull(' ', '%s', $turnier->Dancetype->name)), 0, 0, 'L');
				$this->Cell(0, 4, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull(' ', '%s', $turnier->bemerkung)), 0, 1, 'L');
			}
			$this->Ln(2);
			$done = false;
			foreach ($team->getAssignments(\Assignment::STATE_CONFIRMED) as $einsatz) {
				$this->SetFont('Arial', 'U', 8);
				if (!$done) {
					$print = 'Wertungsrichter';
					$done = true;
				} else {
					$print = '';
				}
				$this->Cell(25, 4, $print, 0, 0, 'L');
				$this->SetFont('Arial', '', 8);
				$wr = $einsatz->User;
				if ($wr->title != '') {
					$titel = $wr->title . ' ';
				} else {
					$titel = '';
				}
				$this->Cell(60, 4, $this->utf8ToLatin1($wr->getAnrede() . ' ' . $titel . $wr->name . ' ' . $wr->surname . ' (' . $wr->Club->name . ') [' . $wr->Club->LTV->name . ']'), 0, 1, 'L');
				$this->SetFont('Arial', '', 6);
				if($wr->dtvnr && $wr->getAsDTVNummer('dtvnr')->praefix == 'DE') {
					$this->Cell(25, 3, '', 0, 0, 'L');
					$this->Cell(0, 3, 'DTV-Nummer: ' . $wr->dtvnr, 0, 1, 'L');
				}
				if ($this->getConfigValue('zwe_use_addresses')) {
					$this->Cell(25, 3, '', 0, 0, 'L');
					$addr = 'Adresse: ' . \ZWE\StringUtils::formatIfnull('', '%s, ', $wr->strasse) . \ZWE\StringUtils::formatIfnull('', '%s', $wr->plz . ' ' . $wr->ort);
					$this->Cell(0, 3, $this->utf8ToLatin1($addr), 0, 1, 'L');
				}
				$this->Cell(25, 3, '', 0, 0, 'L');
				$phonenumbers = '';
				if (!\ZWE\StringUtils::isnullString($wr->phone)) {
					$phonenumbers .= $wr->phone;
				}
				if (!\ZWE\StringUtils::isnullString($wr->mobile)) {
					$phonenumbers = \ZWE\StringUtils::formatIfnull('', '%s, ', $phonenumbers) . $wr->mobile;
				}
				$this->Cell(0, 3, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull('', 'Telefon: %s', $phonenumbers)), 0, 1, 'L');
				$this->Cell(25, 3, '', 0, 0, 'L');
				$this->Cell(0, 3, $this->utf8ToLatin1(\ZWE\StringUtils::formatIfnull('', 'E-Mail: %s', $wr->email)), 0, 1, 'L');
				$this->Cell(25, 3, '', 0, 0, 'L');
				$licenseeffective = $wr->LicenseEffective->id == 6 ? '' : $wr->LicenseEffective->shortname;
				$this->Cell(0, 3, $this->utf8ToLatin1(sprintf('Lizenz: Standard %s / Latein %s', $wr->LicenseSt->shortname, $wr->LicenseLat->shortname) . \ZWE\StringUtils::formatIfnull('', ' (nur bis Startklasse %s)', $licenseeffective)), 0, 1, 'L');
			}
		}
		$this->SetCompression(true);
	}

	public function Footer()
	{
		try {
			$footerFile = $this->resourceStorage->retrieve('pdf_footer.png');
			$this->Image($footerFile->getAbsolutePath(), 0, 268.6, 210);
		} catch (FileNotFoundException $e) {
		}
	}

	public function Header()
	{
		try {
			$headerFile = $this->resourceStorage->retrieve('pdf_header.png');
			$this->Image($headerFile->getAbsolutePath(), 0, 0, 210);
		} catch (FileNotFoundException $e) {
		}
	}

	public function saveAs(File $file)
	{
		$this->Output($file->getAbsolutePath(), 'F');
	}

	public function send()
	{
		$this->Output($this->getOutputFilename(), 'D');
	}

	public function view()
	{
		$this->Output($this->getOutputFilename(), 'I');
	}

	private function getConfigValue($name)
	{
		return $this->settingsRegistry->get($name)->getValue();
	}

	private function utf8ToLatin1($input)
	{
		$targetEncoding = version_compare(PHP_VERSION, '8.0', '>') ? 'ISO-8859-1//IGNORE' : ($this->translitAvailable ? 'ISO-8859-1//TRANSLIT' : 'ISO-8859-1');
		if ($input) {
			return iconv('UTF-8', $targetEncoding, removeSpecialChars($input));
		}
		return null;
	}
}
