<?php
namespace ZWE\Export;

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xls;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use ZWE\Container\ZWEApplicationContainer;
use ZWE\FileStorage\File;

abstract class AbstractExcelExport
{

	const FORMAT_XLSX = "xlsx";
	const FORMAT_XLS = "xls";

	/**
	 * @var ZWEApplicationContainer
	 */
	protected $container;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Spreadsheet
	 */
	protected $excel;

	/**
	 * @var bool
	 */
	protected $created = false;

	public function __construct(ZWEApplicationContainer $container)
	{
		$this->container = $container;
		$this->excel = new Spreadsheet();
		if ($this->excel->getActiveSheet()) {
			$this->excel->removeSheetByIndex(0);
		}
	}

	public function saveAs(File $file, $format = AbstractExcelExport::FORMAT_XLSX)
	{
		$this->checkCreated();
		$writer = $this->createWriter($format);
		$writer->save($file->getAbsolutePath());
	}

	/**
	 * @codeCoverageIgnore
	 */
	public function send($format = self::FORMAT_XLSX, $baseFilename = null)
	{
		if ($baseFilename == null) {
			throw new \InvalidArgumentException("Basis-Dateiname erforderlich!");
		}

		$this->checkCreated();
		$writer = $this->createWriter($format);

		switch ($format) {
			case self::FORMAT_XLS:
				header('Content-type: application/vnd.ms-excel', true);
				header('Content-disposition: attachment; filename="' . $baseFilename . '.xls"', true);
				break;
			case self::FORMAT_XLSX:
			default:
				header('Content-type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', true);
				header('Content-disposition: attachment; filename="' . $baseFilename . '.xlsx"', true);
				break;
		}
		header('Cache-control: private, max-age: 0, must-revalidate', true);
		$writer->save('php://output');
	}

	/**
	 * @return bool
	 */
	public static function isXlsxAvailable()
	{
		return extension_loaded('zip');
	}

	/**
	 * @return array
	 */
	public static function getAvailableFormats()
	{
		if (self::isXlsxAvailable()) {
			return array(self::FORMAT_XLSX => "Excel 2007 (.xlsx)", self::FORMAT_XLS => "Excel 97 (.xls)");
		} else {
			return array(self::FORMAT_XLS => "Excel 97 (.xls)");
		}
	}

	/**
	 * @return string
	 */
	public static function getDefaultFormat()
	{
		if (self::isXlsxAvailable()) {
			return self::FORMAT_XLSX;
		} else {
			return self::FORMAT_XLS;
		}
	}

	protected function checkCreated()
	{
		if (!$this->created) {
			throw new \RuntimeException("Dokument ist noch nicht erstellt!");
		}
	}

	/**
	 * @param $format
	 * @return \PhpOffice\PhpSpreadsheet\Writer\BaseWriter
	 */
	protected function createWriter($format)
	{
		switch ($format) {
			case self::FORMAT_XLS:
				return new Xls($this->excel);
			case self::FORMAT_XLSX:
			default:
				return new Xlsx($this->excel);
		}
	}
}
