<?php
namespace ZWE\ESV;

/**
 * DTV-Nummer.
 *
 * @package ZWE\ESV
 * @property-read string $praefix Präfix
 * @property-read int $nummer Nummer
 * @property-read int $pz Prüfziffer
 */
class DTVNummer implements \JsonSerializable
{

	private $praefix;
	private $nummer;
	private $pz;

	public static function create($praefix, $nummer)
	{
		return new DTVNummer($praefix, $nummer);
	}

	private function __construct($praefix, $nummer, $pz = null)
	{
		if (strlen(trim($praefix)) != 2) {
			throw new \InvalidArgumentException('Präfix muss zwei Zeichen lang sein!');
		}
		if ($nummer < 0) {
			throw new \InvalidArgumentException('Nummer muss größer 0 sein!');
		}
		$this->praefix = strtoupper(trim($praefix));
		$this->nummer = intval($nummer);
		if (!$pz) {
			$this->pz = $this->pruefzifferErmitteln($this->nummer);
		} else {
			$this->pz = intval($pz);
		}
	}

	public function __get($key)
	{
		// @codeCoverageIgnoreStart
		switch ($key) {
			case 'praefix':
			case 'nummer':
			case 'pz':
				return $this->$key;
			default:
				throw new \RuntimeException("Property {$key} existiert nicht!");
		}
		// @codeCoverageIgnoreEnd
	}

	public function __isset($key)
	{
		// @codeCoverageIgnoreStart
		switch ($key) {
			case 'praefix':
			case 'nummer':
			case 'pz':
				return isset($this->$key);
			default:
				return false;
		}
		// @codeCoverageIgnoreEnd
	}

	public function isValid()
	{
		return $this->pz == $this->pruefzifferErmitteln($this->nummer);
	}

	public function jsonSerialize()
	{
		return $this->__toString();
	}

	public function __toString()
	{
		return "{$this->praefix}{$this->nummer}{$this->pz}";
	}

	public static function fromString($input)
	{
		$matches = array();
		if (!preg_match('/^([a-zA-Z]{2})(\d+)(\d)$/', trim($input), $matches)) {
			throw new \InvalidArgumentException("{$input} hat keine gültige Syntax!");
		}
		return new DTVNummer($matches[1], $matches[2], $matches[3]);
	}

	private function pruefzifferErmitteln($nummer)
	{
		$nummerAsString = "{$nummer}";
		$sum = 0;
		$k = 2;
		for ($i = strlen($nummerAsString) - 1; $i >= 0; $i--) {
			$sum += $k * intval($nummerAsString[$i]);
			$k++;
		}
		$diff11 = 11 - ($sum % 11);
		if ($diff11 == 10) {
			return 0;
		} elseif ($diff11 == 11) {
			return 5;
		} else {
			return $diff11;
		}
	}

}