<?php
namespace ZWE\DateStorage;

class DateStorage
{

	/**
	 * @param \DateTime $dateTime
	 * @param string $group Speichergruppe
	 * @param string $id ID des persistenten Datum oder <code>NULL</code> (ID wird generiert)
	 * @return \PersistentDate
	 * @throws DateAlreadyExistsException falls das persistente Datum mit der angegebenen ID bereits existiert
	 * @throws \InvalidArgumentException falls keine Speichergruppe angegeben wurde
	 */
	public function addNewPersistentDate(\DateTime $dateTime, $group, $id = null)
	{
		if ($group == null) {
			throw new \InvalidArgumentException("Speichergruppe erforderlich!");
		}
		if ($id == null) {
			$persistentDate = \PersistentDate::newWithGeneratedId($group);
		} else {
			$existing = $this->findPersistenDate($id);
			if ($existing != null) {
				throw new DateAlreadyExistsException('Es existiert bereits ein gespeichertes Datum mit ID ' . $id);
			}
			$persistentDate = \PersistentDate::newInstance($id, $group);
		}
		$persistentDate->setValue($dateTime);
		$persistentDate->save();
		return $persistentDate;
	}

	/**
	 * L&ouml;scht ein persistentes Datum.
	 *
	 * @param $id ID des persistenten Datum
	 * @return bool gel&ouml;scht ja/nein
	 */
	public function deletePersistentDate($id)
	{
		$pd = \Doctrine::getTable('PersistentDate')->find($id);
		if ($pd) {
			$pd->delete();
			return true;
		}
		return false;
	}

	/**
	 * L&ouml;scht abgelaufene Daten in einer Speichergruppe.
	 *
	 * @param $group Speichergruppe
	 */
	public function deleteExpiredDatesInGroup($group)
	{
		$query = \Doctrine_Query::create()
			->delete('PersistentDate pd')
			->where('pd.persistent_date < NOW() - INTERVAL 1 DAY')
			->andWhere('pd.dategroup = ?', $group);
		$query->execute();
	}

	/**
	 * L&auml;dt ein persistentes Datum.
	 *
	 * @param $id ID des persistenten Datums
	 * @return \PersistentDate
	 * @throws DateStorageException falls das persitente Datum nicht existiert
	 */
	public function loadPersistentDate($id)
	{
		$persistentDate = $this->findPersistenDate($id);
		if ($persistentDate != null) {
			return $persistentDate;
		} else {
			throw new DateStorageException('Gespeichertes Datum mit ID ' . $id . ' wurde nicht gefunden!');
		}
	}

	/**
	 * Findet ein persistentes Datum.
	 *
	 * @param $id ID des persistenten Datums
	 * @return \PersistentDate|null
	 */
	public function findPersistenDate($id)
	{
		$query = \Doctrine_Query::create()
			->select('dp.*')
			->from('PersistentDate dp')
			->where('dp.id = ?', $id);
		return $query->fetchOne();
	}

	/**
	 * L&auml;dt alle Daten eines Speichergruppe.
	 *
	 * @param $group
	 * @return \Doctrine_Collection
	 */
	public function loadPersistentDateGroup($group)
	{
		$query = \Doctrine_Query::create()
			->select('dp.*')
			->from('PersistentDate dp')
			->where('dp.dategroup = ?', $group)
			->orderBy('dp.persistent_date ASC');
		return $query->execute();
	}

}