<?php

namespace ZWE\Container;

use Doctrine\Common\Annotations\AnnotationReader;
use Doctrine\Common\Annotations\AnnotationRegistry;
use JMS\Serializer\Serializer;
use JMS\Serializer\SerializerBuilder;
use Metadata\MetadataFactoryInterface;
use Rych\Random\Random;
use Symfony\Component\Translation\Loader\XliffFileLoader;
use Symfony\Component\Translation\Translator;
use Symfony\Component\Validator\Mapping\Factory\LazyLoadingMetadataFactory;
use Symfony\Component\Validator\Mapping\Loader\AnnotationLoader;
use Symfony\Component\Validator\Validation;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Tanzsport\ESV\API\Client;
use Tanzsport\ESV\API\Endpunkt;
use ZWE\ESV\DTVNummerGenerator;
use ZWE\ESV\ESVService;
use ZWE\ESV\ExportFormat\ExportZWEX;
use ZWE\Export\Einsatzplan\ExportEinsatzplanPDF;
use ZWE\Export\Einsatzplan\ExportEinsatzplanPDFImpl;
use ZWE\Export\Freigaben\ExportFreigabenExcel;
use ZWE\Export\Freigaben\ExportFreigabenExcelImpl;
use ZWE\Export\Freigaben\ExportPlanungExcel;
use ZWE\Export\Freigaben\ExportPlanungExcelImpl;
use ZWE\FileStorage\EventAttachmentStorageImpl;
use ZWE\FileStorage\FileStorage;
use ZWE\FileStorage\FileStorageImpl;
use ZWE\Images\ImageResourceService;
use ZWE\Jobs\JobFactory;
use ZWE\Mail\MailingService;
use ZWE\Mail\MailTextService;
use ZWE\Mail\MessageCreator;
use ZWE\Queue\DatabaseQueueDriver;
use ZWE\Queue\Queue;
use ZWE\Queue\QueueDriver;
use ZWE\Queue\SyncQueueDriver;
use ZWE\Repository\AssignmentplanRepository;
use ZWE\Repository\AssignmentRepository;
use ZWE\Repository\AvailabilityRepository;
use ZWE\Repository\ClubRepository;
use ZWE\Repository\CompetitionRepository;
use ZWE\Repository\CountryRepository;
use ZWE\Repository\DancetypeRepository;
use ZWE\Repository\EmailTextRepository;
use ZWE\Repository\EventRepository;
use ZWE\Repository\LicenseRepository;
use ZWE\Repository\LocationRepository;
use ZWE\Repository\LTVRepository;
use ZWE\Repository\RoleRepository;
use ZWE\Repository\StartclassRepository;
use ZWE\Repository\StartgroupRepository;
use ZWE\Repository\UserRepository;
use ZWE\Repository\WRTeamRepository;
use ZWE\Settings\SettingsRegistry;
use ZWE\User\PasswordResetService;

abstract class ZWEApplicationContainerImpl implements ZWEApplicationContainer
{
	const CONFIGSOURCE = "configurationSource";

	const ASSIGNMENTPLAN_STORAGE = "assignmentplanStorage";
	const RESOURCE_STORAGE = "resourceStorage";
	const TEMP_STORAGE = "tempStorage";
	const WORKSPACE_STORAGE = "workspaceStorage";

	const SETTINGS_SERVICE = "settingsService";
	const MESSAGE_CREATOR = "messageCreator";
	const MAILING_SERVICE = "mailingService";
	const PWRESET_SERVICE = "pwResetService";
	const MAILTEXT_SERVICE = "mailtextService";
	const IMG_SERVICE = "imgService";
	const DTVNUMMERGENERATOR_SERVICE = "dtvNummerGeneratorService";
	const ESVCLIENT_SERVICE = "esvClientService";
	const ESV_SERVICE = "esvService";
	const SERIALIZER_SERVICE = "serializerService";
	const TRANSLATOR_SERVICE = "translatorService";
	const ANNOTATION_READER = "annotationReader";
	const ANNOTATION_LOADER = "annotationLoader";
	const METADATA_FACTORY = "metadataFactory";
	const VALIDATOR_SERVICE = "validatorService";
	const RANDOM_SERVICE = "randomService";
	const LOGGING_SERVICE = "loggingService";
	const QUEUE = "queue";
	const QUEUE_DRIVER = "queueDriver";
	const JOB_FACTORY = "jobFactory";

	const PLAN_REPO = "assPlanRepo";
	const ASS_REPO = "assRepo";
	const AV_REPO = "avRepo";
	const CLUB_REPO = "clubRepo";
	const COMP_REPO = "compRepo";
	const TYPE_REPO = "typeRepo";
	const EVENT_REPO = "eventRepo";
	const LICENSE_REPO = "licenseRepo";
	const LOCATION_REPO = "locationRepo";
	const LTV_REPO = "ltvRepo";
	const ROLE_REPO = "roleRepo";
	const CLASS_REPO = "classRepo";
	const GROUP_REPO = "groupRepo";
	const USER_REPO = "userRepo";
	const TEAM_REPO = "teamRepo";
	const MAILTEXT_REPO = "mailtextRepo";
	const COUNTRY_REPO = "countryRepo";

	/**
	 * @var ZWEApplicationContainer
	 */
	private static $instance;

	/**
	 * @var \Pimple
	 */
	private $container;

	private $requiredExtensions = array(
		'gd', 'iconv', 'json', 'libxml', 'PDO', 'pdo_mysql', 'xml', 'zlib'
	);

	private $optionalExtensions = array(
		'zip'
	);

	public function __construct()
	{
		self::$instance = $this;
		$this->checkEnvironment();
		$this->container = new \Pimple();
		$this->setUp();
	}

	/**
	 * @return ZWEApplicationContainer
	 */
	public static function getInstance()
	{
		return self::$instance;
	}

	protected function setUp()
	{
		$this->addConfigurationSource();

		$this->createDirectories();
		$this->addAssignmentplanStorage();
		$this->addResourceStorage();
		$this->addTempStorage();
		$this->addWorkspaceStorage();

		$this->addAssignmentplanRepo();
		$this->addAssignmentRepo();
		$this->addAvailabilityRepo();
		$this->addClubRepo();
		$this->addCompetitionRepo();
		$this->addDancetypeRepo();
		$this->addEventRepo();
		$this->addLicenseRepo();
		$this->addLocationRepo();
		$this->addLTVRepo();
		$this->addRoleRepo();
		$this->addStartclassRepo();
		$this->addStartgroupRepo();
		$this->addUserRepo();
		$this->addWRTeamRepo();
		$this->addMailTextRepo();
		$this->addCountryRepo();

		$this->addSettingsService();
		$this->addMessageCreator();
		$this->addMailingService();
		$this->addPasswordResetService();
		$this->addMailTextService();
		$this->addImgService();
		$this->addDtvNummerGeneratorService();
		$this->addEsvClient();
		$this->addEsvService();
		$this->addSerializer();
		$this->addTranslator();
		$this->addAnnotationReader();
		$this->addAnnotationLoader();
		$this->addMetadataFactory();
		$this->addValidator();
		$this->addRandom();
		$this->addLoggingService();
		$this->addQueue();
		$this->addQueueDriver();
		$this->addJobFactory();
	}

	/* === ADD CONFIGURATION SOURCE === */

	protected abstract function addConfigurationSource();

	/* === ADD STORAGE === */

	private function createDirectories()
	{
		$directories = array('files', 'logs', 'resources', 'tmp', 'worker');
		$workspace = $this->getConfigurationSource()->offsetGet('zwe.workspace');
		if (!is_dir($workspace)) {
			throw new \RuntimeException("Workspace-Pfad existiert nicht!");
		}
		foreach ($directories as $directory) {
			if (!is_dir($workspace . '/' . $directory)) {
				if (!@mkdir($workspace . '/' . $directory)) {
					throw new \RuntimeException("Verzeichnis {$directory} konnte nicht erstellt werden!");
				}
			}
		}
	}

	protected function addAssignmentplanStorage()
	{
		$this->addSingletonBean(self::ASSIGNMENTPLAN_STORAGE, function () {
			return new FileStorageImpl(ZWEApplicationContainerImpl::getInstance()->getConfigurationSource(), 'files');
		});
	}

	protected function addResourceStorage()
	{
		$this->addSingletonBean(self::RESOURCE_STORAGE, function () {
			return new FileStorageImpl(ZWEApplicationContainerImpl::getInstance()->getConfigurationSource(), 'resources');
		});
	}

	protected function addTempStorage()
	{
		$this->addSingletonBean(self::TEMP_STORAGE, function () {
			return new FileStorageImpl(ZWEApplicationContainerImpl::getInstance()->getConfigurationSource(), 'tmp');
		});
	}

	protected function addWorkspaceStorage()
	{
		$this->addSingletonBean(self::WORKSPACE_STORAGE, function () {
			return new FileStorageImpl(ZWEApplicationContainerImpl::getInstance()->getConfigurationSource(), '');
		});
	}

	/* === ADD REPOSITORIES === */

	protected function addAssignmentplanRepo()
	{
		$this->addSingletonBean(self::PLAN_REPO, function () {
			return new AssignmentplanRepository();
		});
	}

	protected function addAssignmentRepo()
	{
		$this->addSingletonBean(self::ASS_REPO, function () {
			return new AssignmentRepository();
		});
	}

	protected function addAvailabilityRepo()
	{
		$this->addSingletonBean(self::AV_REPO, function () {
			return new AvailabilityRepository();
		});
	}

	protected function addClubRepo()
	{
		$this->addSingletonBean(self::CLUB_REPO, function () {
			return new ClubRepository();
		});
	}

	protected function addCompetitionRepo()
	{
		$this->addSingletonBean(self::COMP_REPO, function () {
			return new CompetitionRepository();
		});
	}

	protected function addDancetypeRepo()
	{
		$this->addSingletonBean(self::TYPE_REPO, function () {
			return new DancetypeRepository();
		});
	}

	protected function addEventRepo()
	{
		$this->addSingletonBean(self::EVENT_REPO, function () {
			return new EventRepository();
		});
	}

	protected function addLicenseRepo()
	{
		$this->addSingletonBean(self::LICENSE_REPO, function () {
			return new LicenseRepository();
		});
	}

	protected function addLocationRepo()
	{
		$this->addSingletonBean(self::LOCATION_REPO, function () {
			return new LocationRepository();
		});
	}

	protected function addLTVRepo()
	{
		$this->addSingletonBean(self::LTV_REPO, function () {
			return new LTVRepository();
		});
	}

	protected function addRoleRepo()
	{
		$this->addSingletonBean(self::ROLE_REPO, function () {
			return new RoleRepository();
		});
	}

	protected function addStartclassRepo()
	{
		$this->addSingletonBean(self::CLASS_REPO, function () {
			return new StartclassRepository();
		});
	}

	protected function addStartgroupRepo()
	{
		$this->addSingletonBean(self::GROUP_REPO, function () {
			return new StartgroupRepository();
		});
	}

	protected function addUserRepo()
	{
		$this->addSingletonBean(self::USER_REPO, function () {
			return new UserRepository();
		});
	}

	protected function addWRTeamRepo()
	{
		$this->addSingletonBean(self::TEAM_REPO, function () {
			return new WRTeamRepository();
		});
	}

	protected function addMailTextRepo()
	{
		$this->addSingletonBean(self::MAILTEXT_REPO, function () {
			return new EmailTextRepository();
		});
	}

	protected function addCountryRepo()
	{
		$this->addSingletonBean(self::COUNTRY_REPO, function () {
			return new CountryRepository();
		});
	}

	/* === ADD SERVICES === */

	protected function addSettingsService()
	{
		$this->addSingletonBean(self::SETTINGS_SERVICE, function () {
			return new SettingsRegistry();
		});
	}

	protected function addMessageCreator()
	{
		$this->addSingletonBean(self::MESSAGE_CREATOR, function () {
			return new MessageCreator(ZWEApplicationContainerImpl::getInstance(), ZWEApplicationContainerImpl::getInstance()->isLoadInternalMessageHelpers());
		});
	}

	protected function addMailingService()
	{
		$this->addSingletonBean(self::MAILING_SERVICE, function () {
			return new MailingService(ZWEApplicationContainerImpl::getInstance());
		});
	}

	protected function addPasswordResetService()
	{
		$this->addSingletonBean(self::PWRESET_SERVICE, function () {
			return new PasswordResetService(ZWEApplicationContainerImpl::getInstance());
		});
	}

	protected function addMailTextService()
	{
		$this->addSingletonBean(self::MAILTEXT_SERVICE, function () {
			return new MailTextService(ZWEApplicationContainerImpl::getInstance());
		});
	}

	protected function addImgService()
	{
		$this->addSingletonBean(self::IMG_SERVICE, function () {
			return new ImageResourceService(ZWEApplicationContainerImpl::getInstance());
		});
	}

	protected function addDtvNummerGeneratorService()
	{
		$this->addSingletonBean(self::DTVNUMMERGENERATOR_SERVICE, function () {
			return new DTVNummerGenerator();
		});
	}

	protected function addEsvClient()
	{
		$this->addSingletonBean(self::ESVCLIENT_SERVICE, function () {
			return new Client(new Endpunkt(Endpunkt::PROD), 'Web-ZWE/3.0', '1fb3fa0f74b9ee2dc65a9de6ea92cd95', 'Web-ZWE', 'J8Gk6mNh', true, false);
		});
	}

	protected function addEsvService()
	{
		$this->addSingletonBean(self::ESV_SERVICE, function () {
			return new ESVService(
				ZWEApplicationContainerImpl::getInstance()->getEsvClient(),
				ZWEApplicationContainerImpl::getInstance()->getTempStorage(),
				ZWEApplicationContainerImpl::getInstance()->getDTVNummerGenerator(),
				ZWEApplicationContainerImpl::getInstance()->getUserRepo(),
				ZWEApplicationContainerImpl::getInstance()->getClubRepo(),
				ZWEApplicationContainerImpl::getInstance()->getLTVRepo(),
				ZWEApplicationContainerImpl::getInstance()->getLicenseRepo(),
				ZWEApplicationContainerImpl::getInstance()->getCountryRepo(),
				ZWEApplicationContainerImpl::getInstance()->getSerializer()
			);
		});
	}

	protected function addSerializer()
	{
		$this->addSingletonBean(self::SERIALIZER_SERVICE, function () {
			AnnotationRegistry::registerLoader('class_exists');
			return SerializerBuilder::create()->build();
		});
	}

	protected function addTranslator()
	{
		$this->addSingletonBean(self::TRANSLATOR_SERVICE, function () {
			return new Translator('de_DE');
		});
	}

	protected function addAnnotationReader()
	{
		$this->addSingletonBean(self::ANNOTATION_READER, function () {
			AnnotationRegistry::registerLoader('class_exists');
			return new AnnotationReader();
		});
	}

	protected function addAnnotationLoader()
	{
		$this->addSingletonBean(self::ANNOTATION_LOADER, function () {
			return new AnnotationLoader($this->getAnnotationReader());
		});
	}

	protected function addMetadataFactory()
	{
		$this->addSingletonBean(self::METADATA_FACTORY, function () {
			return new LazyLoadingMetadataFactory($this->getAnnotationLoader());
		});
	}

	protected function addValidator()
	{
		$this->addSingletonBean(self::VALIDATOR_SERVICE, function () {
			$this->getTranslator()->addLoader('xlf', new XliffFileLoader());
			if ($this->isInstalledInVendorDir()) {
				$this->getTranslator()->addResource('xlf', $this->getVendorDir() . '/symfony/validator/Resources/translations/validators.de.xlf', 'de_DE', 'validators');
			} else {
				$this->getTranslator()->addResource('xlf', __DIR__ . '/../../../../vendor/symfony/validator/Resources/translations/validators.de.xlf', 'de_DE', 'validators');
			}
			return Validation::createValidatorBuilder()
				->setTranslator($this->getTranslator())
				->setTranslationDomain('validators')
				->setMetadataFactory($this->getMetadataFactory())
				->getValidator();
		});
	}

	protected function addRandom()
	{
		$this->addSingletonBean(self::RANDOM_SERVICE, function () {
			return new Random();
		});
	}

	protected function addLoggingService()
	{
		$this->addSingletonBean(self::LOGGING_SERVICE, function () {
			return new LoggingServiceImpl($this);
		});
	}

	protected function addQueue() {
		$this->addSingletonBean(self::QUEUE, function() {
			return new Queue($this, $this->getQueueDriver());
		});
	}

	protected function addQueueDriver() {
		$this->addSingletonBean(self::QUEUE_DRIVER, function() {
			$useBackgroundQueue = $this->getSettingsService()->get('zwe_use_background_queue')->getValue();
			if ($useBackgroundQueue) {
				return new DatabaseQueueDriver($this->getValidator(), $this->getLoggingService());
			} else {
				return new SyncQueueDriver($this->getValidator(), $this->getLoggingService());
			}
		});
	}

	protected function addJobFactory() {
		$this->addSingletonBean(self::JOB_FACTORY, function() {
			return new JobFactory($this);
		});
	}

	/* === CONTAINER FUNCTIONS === */

	protected function addSingletonBean($name, \Closure $closure)
	{
		// @codeCoverageIgnoreStart
		if ($name == null) {
			throw new \InvalidArgumentException("Bean name required!");
		}
		// @codeCoverageIgnoreEnd
		$this->container[$name] = $closure;
	}

	protected function getSingletonBean($name)
	{
		// @codeCoverageIgnoreStart
		if ($name == null) {
			throw new \InvalidArgumentException("Bean name required!");
		}
		// @codeCoverageIgnoreEnd
		return $this->container[$name];
	}

	/* === CHECK ENVIRONMENT === */

	private function checkEnvironment()
	{
		$this->checkPhpVersion();
		$this->checkExtensions();
	}

	private function checkPhpVersion()
	{
		// @codeCoverageIgnoreStart
		if (version_compare(PHP_VERSION, '5.4.0') < 0) {
			throw new \RuntimeException("PHP-Version >= 5.4.0 erforderlich!");
		}
		// @codeCoverageIgnoreEnd
	}

	private function checkExtensions()
	{
		foreach ($this->requiredExtensions as $requiredExtension) {
			// @codeCoverageIgnoreStart
			if (!extension_loaded($requiredExtension)) {
				throw new \RuntimeException("PHP-Extension {$requiredExtension} erforderlich!");
			}
			// @codeCoverageIgnoreEnd
		}
	}

	/* === GET CONFIGURATIONSOURCE === */

	/**
	 * @return ZWEConfigurationSource
	 */
	public function getConfigurationSource()
	{
		return $this->getSingletonBean(self::CONFIGSOURCE);
	}

	/* === GET STORAGE === */

	/**
	 * @return \ZWE\FileStorage\FileStorage
	 */
	public function getAssignmentplanStorage()
	{
		return $this->getSingletonBean(self::ASSIGNMENTPLAN_STORAGE);
	}

	/**
	 * @return \ZWE\FileStorage\FileStorage
	 */
	public function getResourceStorage()
	{
		return $this->getSingletonBean(self::RESOURCE_STORAGE);
	}

	/**
	 * @return \ZWE\FileStorage\FileStorage
	 */
	public function getTempStorage()
	{
		return $this->getSingletonBean(self::TEMP_STORAGE);
	}

	public function getWorkspaceStorage()
	{
		return $this->getSingletonBean(self::WORKSPACE_STORAGE);
	}

	/* === GET REPOSITORIES === */

	/**
	 * @return AssignmentplanRepository
	 */
	public function getAssignmentplanRepo()
	{
		return $this->getSingletonBean(self::PLAN_REPO);
	}

	/**
	 * @return AssignmentRepository
	 */
	public function getAssignmentRepo()
	{
		return $this->getSingletonBean(self::ASS_REPO);
	}

	/**
	 * @return AvailabilityRepository
	 */
	public function getAvailabilityRepo()
	{
		return $this->getSingletonBean(self::AV_REPO);
	}

	/**
	 * @return ClubRepository
	 */
	public function getClubRepo()
	{
		return $this->getSingletonBean(self::CLUB_REPO);
	}

	/**
	 * @return CompetitionRepository
	 */
	public function getCompetitionRepo()
	{
		return $this->getSingletonBean(self::COMP_REPO);
	}

	/**
	 * @return DancetypeRepository
	 */
	public function getDancetypeRepo()
	{
		return $this->getSingletonBean(self::TYPE_REPO);
	}

	/**
	 * @return EventRepository
	 */
	public function getEventRepo()
	{
		return $this->getSingletonBean(self::EVENT_REPO);
	}

	/**
	 * @return LicenseRepository
	 */
	public function getLicenseRepo()
	{
		return $this->getSingletonBean(self::LICENSE_REPO);
	}

	/**
	 * @return LocationRepository
	 */
	public function getLocationRepo()
	{
		return $this->getSingletonBean(self::LOCATION_REPO);
	}

	/**
	 * @return LTVRepository
	 */
	public function getLTVRepo()
	{
		return $this->getSingletonBean(self::LTV_REPO);
	}

	/**
	 * @return RoleRepository
	 */
	public function getRoleRepo()
	{
		return $this->getSingletonBean(self::ROLE_REPO);
	}

	/**
	 * @return StartclassRepository
	 */
	public function getStartclassRepo()
	{
		return $this->getSingletonBean(self::CLASS_REPO);
	}

	/**
	 * @return StartgroupRepository
	 */
	public function getStartgroupRepo()
	{
		return $this->getSingletonBean(self::GROUP_REPO);
	}

	/**
	 * @return UserRepository
	 */
	public function getUserRepo()
	{
		return $this->getSingletonBean(self::USER_REPO);
	}

	/**
	 * @return WRTeamRepository
	 */
	public function getWRTeamRepo()
	{
		return $this->getSingletonBean(self::TEAM_REPO);
	}

	/**
	 * @return EmailTextRepository
	 */
	public function getEmailTextRepo()
	{
		return $this->getSingletonBean(self::MAILTEXT_REPO);
	}

	public function getCountryRepo()
	{
		return $this->getSingletonBean(self::COUNTRY_REPO);
	}

	// === SERVICES === */

	/**
	 * @return SettingsRegistry
	 */
	public function getSettingsService()
	{
		return $this->getSingletonBean(self::SETTINGS_SERVICE);
	}

	/**
	 * @return MessageCreator
	 */
	public function getMessageCreator()
	{
		return $this->getSingletonBean(self::MESSAGE_CREATOR);
	}

	/**
	 * @return MailingService
	 */
	public function getMailingService()
	{
		return $this->getSingletonBean(self::MAILING_SERVICE);
	}

	/**
	 * @return PasswordResetService
	 */
	public function getPasswordResetService()
	{
		return $this->getSingletonBean(self::PWRESET_SERVICE);
	}

	/**
	 * @return MailTextService
	 */
	public function getMailTextService()
	{
		return $this->getSingletonBean(self::MAILTEXT_SERVICE);
	}

	/**
	 * @return ImageResourceService
	 */
	public function getImageResourceService()
	{
		return $this->getSingletonBean(self::IMG_SERVICE);
	}

	/**
	 * @return DTVNummerGenerator
	 */
	public function getDTVNummerGenerator()
	{
		return $this->getSingletonBean(self::DTVNUMMERGENERATOR_SERVICE);
	}

	/**
	 * @return Client
	 */
	public function getEsvClient()
	{
		return $this->getSingletonBean(self::ESVCLIENT_SERVICE);
	}

	/**
	 * @return ESVService
	 */
	public function getEsvService()
	{
		return $this->getSingletonBean(self::ESV_SERVICE);
	}

	/**
	 * @return Serializer
	 */
	public function getSerializer()
	{
		return $this->getSingletonBean(self::SERIALIZER_SERVICE);
	}

	/**
	 * @return Translator
	 */
	public function getTranslator()
	{
		return $this->getSingletonBean(self::TRANSLATOR_SERVICE);
	}

	/**
	 * @return AnnotationReader
	 */
	public function getAnnotationReader()
	{
		return $this->getSingletonBean(self::ANNOTATION_READER);
	}

	/**
	 * @return AnnotationLoader
	 */
	public function getAnnotationLoader()
	{
		return $this->getSingletonBean(self::ANNOTATION_LOADER);
	}

	/**
	 * @return MetadataFactoryInterface
	 */
	public function getMetadataFactory()
	{
		return $this->getSingletonBean(self::METADATA_FACTORY);
	}

	/**
	 * @return ValidatorInterface
	 */
	public function getValidator()
	{
		return $this->getSingletonBean(self::VALIDATOR_SERVICE);
	}

	/**
	 * @return Random
	 */
	public function getRandom()
	{
		return $this->getSingletonBean(self::RANDOM_SERVICE);
	}

	/**
	 * @return LoggingService
	 */
	public function getLoggingService()
	{
		return $this->getSingletonBean(self::LOGGING_SERVICE);
	}

	/**
	 * @return Queue
	 */
	public function getQueue()
	{
		return $this->getSingletonBean(self::QUEUE);
	}

	/**
	 * @return QueueDriver
	 */
	public function getQueueDriver()
	{
		return $this->getSingletonBean(self::QUEUE_DRIVER);
	}

	/**
	 * @return JobFactory
	 */
	public function getJobFactory()
	{
		return $this->getSingletonBean(self::JOB_FACTORY);
	}

	/* === PDO === */

	/**
	 * @return \PDO
	 */
	public function getPDO()
	{
		return \Doctrine_Manager::connection()->getDbh();
	}

	/* === GET NON-SINGLETONS === */

	/**
	 * @return ExportEinsatzplanPDF
	 */
	public function getNewExportEinsatzplan()
	{
		return new ExportEinsatzplanPDFImpl($this->getResourceStorage(), $this->getSettingsService());
	}

	/**
	 * @return ExportFreigabenExcel
	 */
	public function getNewExportFreigaben()
	{
		return new ExportFreigabenExcelImpl($this);
	}

	/**
	 * @return ExportPlanungExcel
	 */
	public function getNewExportPlanung()
	{
		return new ExportPlanungExcelImpl($this);
	}

	/**
	 * @param \Event $event
	 * @return FileStorage
	 */
	public function getNewEventAttachmentStorage(\Event $event)
	{
		return new EventAttachmentStorageImpl($this->getConfigurationSource(), $event);
	}

	/**
	 * @return ExportZWEX
	 */
	public function getNewExportZWEX()
	{
		return new ExportZWEX($this->getResourceStorage(), $this->getEsvService());
	}

	/* === MISC === */

	/**
	 * @return array
	 */
	public function getRequiredExtensions()
	{
		return $this->requiredExtensions;
	}

	/**
	 * @return array
	 */
	public function getOptionalExtensions()
	{
		return $this->optionalExtensions;
	}

	/**
	 * @return array
	 */
	public function getExtensions()
	{
		return array_merge($this->requiredExtensions, $this->optionalExtensions);
	}

	/**
	 * @return bool
	 */
	public function isInstalledInVendorDir()
	{
		$autoloadFile = __DIR__ . '/../../../../../../autoload.php';
		return file_exists($autoloadFile);
	}

	/**
	 * @return string
	 */
	public function getVendorDir()
	{
		return realpath(__DIR__ . '/../../../../../../');
	}


}
