<?php

namespace ZWE\Cli;

class ZWECli
{

	/**
	 * @var string
	 */
	private $dataFixturesPath;

	/**
	 * @var string
	 */
	private $modelsPath;

	/**
	 * @var string
	 */
	private $modelsGeneratedPath;

	/**
	 * @var string
	 */
	private $migrationsPath;

	/**
	 * @var string
	 */
	private $yamlSchemaPath;

	/**
	 * @var array
	 */
	private $generateModelsOptions;

	private $latestNotification;

	private function __construct()
	{
		$this->dataFixturesPath = __DIR__ . '/../../doctrine/fixtures';
		$this->modelsPath = __DIR__ . '/../../doctrine/model';
		$this->modelsGeneratedPath = $this->modelsPath . '/generated';
		$this->migrationsPath = __DIR__ . '/../../doctrine/migrations';
		$this->yamlSchemaPath = __DIR__ . '/../../doctrine/schema';
		$this->generateModelsOptions = array('generateTableClasses' => false);
	}

	public static function run()
	{
		$cli = new ZWECli();
		$cli->runInternal();
	}

	/**
	 * @param null $version
	 * @return mixed
	 */
	public static function migrate($version = null)
	{
		$cli = new ZWECli();
		return $cli->migrateInternal($version);
	}

	/**
	 * @return mixed
	 */
	public static function loadData()
	{
		$cli = new ZWECli();
		return $cli->loadDataInternal();
	}

	/**
	 * Dispatcher Tasks
	 * @param array $notification
	 */
	public function notify($notification)
	{
		$this->latestNotification = $notification;
	}

	private function getDoctrineConfig()
	{
		$doctrine_config = array(
			'data_fixtures_path' => $this->dataFixturesPath,
			'models_path' => $this->modelsPath,
			'migrations_path' => $this->migrationsPath,
			'yaml_schema_path' => $this->yamlSchemaPath,
			'generate_models_options' => $this->generateModelsOptions
		);
		foreach ($doctrine_config as $key => $value) {
			if (strpos($key, 'path') !== false) {
				if (!file_exists($value)) {
					throw new \RuntimeException("Konfigurierter Pfad für {$key} ({$value}) existiert nicht!");
				}
			}
		}
		return $doctrine_config;
	}

	/**
	 * @param string $version
	 * @return mixed
	 */
	private function migrateInternal($version)
	{
		$task = new \Doctrine_Task_Migrate($this);
		$task->addArgument('migrations_path', $this->migrationsPath);
		if ($version != null) {
			$task->addArgument('version', $version);
		}
		$task->execute();
		return $this->latestNotification;
	}

	/**
	 * @return mixed
	 */
	private function loadDataInternal()
	{
		$task = new \Doctrine_Task_LoadData($this);
		$task->addArgument('data_fixtures_path', $this->dataFixturesPath);
		$task->addArgument('models_path', $this->modelsPath);
		$task->addArgument('append', false);
		$task->execute();
		return $this->latestNotification;
	}

	private function readDsnFromConfig($file)
	{
		if (!is_file($file)) {
			throw new \RuntimeException("Konfigurationsdatei {$file} existiert nicht!");
		}
		if (!is_readable($file)) {
			throw new \RuntimeException("Konfigurationsdatei {$file} ist nicht lesbar!");
		}

		$db = array();
		// needed for inclusion
		define('BASEPATH', __DIR__);
		define('ENVIRONMENT', 'irrelevant');
		require $file;
		if (isset($db['default'])) {
			$default = $db['default'];
			if (isset($default['dsn']) && trim($default['dsn']) != '') {
				return $default['dsn'];
			} else {
				$keys = ['dbdriver', 'username', 'password', 'hostname', 'database'];
				foreach ($keys as $key) {
					if (!isset($default[$key])) {
						throw new \RuntimeException("Konfigurationsoption {$key} fehlt!");
					}
				}
				return sprintf('%1$s://%2$s:%3$s@%4$s/%5$s', $default['dbdriver'], $default['username'], $default['password'], $default['hostname'], $default['database']);
			}
		} else {
			throw new \RuntimeException("Konfigurationsdatei enthält keine DB-Konfiguration");
		}
	}

	private function runInternal()
	{
		$doctrine_config = $this->getDoctrineConfig();

		$options = getopt('d:c:');
		$dsn = null;
		if (isset($options['d'])) {
			$dsn = $options['d'];
		} else if (isset($options['c'])) {
			$dsn = $this->readDsnFromConfig($options['c']);
		}

		if (!$dsn) {
			echo "Verbindungskonfiguration erforderlich; Option -d für DSN oder -c für Pfad zur Konfigurationsdatei!\n";
			exit(1);
		}
		$initializer = new \DoctrineInitializer($dsn, array($this->modelsPath, $this->modelsGeneratedPath));
		$initializer->connect();

		$forwardArgs = [];
		foreach ($_SERVER['argv'] as $key => $value) {
			if ($key == 0) {
				$forwardArgs[] = $value;
			} else if (!in_array($_SERVER['argv'][$key], ['-d', '-c'])) {
				if (!in_array($_SERVER['argv'][$key - 1], ['-d', '-c'])) {
					$forwardArgs[] = $value;
				}
			}
		}

		$cli = new \Doctrine_Cli($doctrine_config);
		$cli->run($forwardArgs);
	}

}
