<?php
namespace ZWE\Calendar;

use ZWE\Repository\AssignmentRepository;
use ZWE\Repository\AvailabilityRepository;

class CalendarSheet
{

	private $year;
	private $month;

	private $assignments = array();
	private $availabilities = array();

	private $asr;
	private $avr;

	private $names = array(
		'long' => array("", "Januar", "Februar", "M&auml;rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember"),
		'short' => array("", "Jan", "Feb", "Mär", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez"),
	);

	private $user;

	private $iter;

	public function __construct($year, $month, $user = null)
	{
		$this->year = $year;
		$this->month = $month;
		$this->avr = new AvailabilityRepository();
		$this->asr = new AssignmentRepository();
		if ($user != null) {
			$this->user = $user;
			$this->loadUserData();
		}
	}

	public function adjust($year, $month, $user = null)
	{
		$this->year = $year;
		$this->month = $month;
		$this->iter = null;
		if ($user != null) {
			$this->user = $user;
			$this->loadUserData();
		}
	}

	public function getIterator()
	{
		if ($this->iter == null) {
			$this->iter = new CalendarSheetIterator($this->year, $this->month, $this->user, $this->availabilities, $this->assignments);
		}
		return $this->iter;
	}

	public function getLastDayInCurrentMonth()
	{
		$last = null;
		foreach ($this->getIterator() as $day) {
			if ($day->isInCurrentMonth()) {
				$last = $day;
			}
		}
		return $last;
	}

	public function getMonth()
	{
		return $this->month;
	}

	public function getMonthAsShortString()
	{
		return $this->names['short'][$this->getMonth()];
	}

	public function getMonthAsString()
	{
		return $this->names['long'][$this->getMonth()];
	}

	public function getYear()
	{
		return $this->year;
	}

	private function loadUserData()
	{
		$assignments = $this->asr->getAssignmentsByUser($this->user, $this->year, $this->month);
		foreach ($assignments as $assignment) {
			$tst = $assignment->WRTeam->getDateTimeObject('date_from')->format('U');
			if (!isset($this->assignments[$tst])) {
				$this->assignments[$tst] = array();
			}
			array_push($this->assignments[$tst], $assignment);
		}
		$availabilities = $this->avr->getAvailabilitiesByUser($this->user, $this->year, $this->month);
		foreach ($availabilities as $availability) {
			$tst = $availability->getDateTimeObject('datum')->format('U');
			$this->availabilities[$tst] = $availability;
		}
	}

	private function nextMonth()
	{
		return $this->month == 12 ? 1 : $this->month + 1;
	}

	public function nextSheet()
	{
		return $this->month == 12 ? new CalendarSheet($this->nextYear(), $this->nextMonth()) : new CalendarSheet($this->year, $this->nextMonth());
	}

	private function nextYear()
	{
		return $this->year + 1;
	}

	public function nextYearSheet()
	{
		return new CalendarSheet($this->nextYear(), 1);
	}

	private function previousMonth()
	{
		return $this->month == 1 ? 12 : $this->month - 1;
	}

	public function previousSheet()
	{
		return $this->month == 1 ? new CalendarSheet($this->previousYear(), $this->previousMonth()) : new CalendarSheet($this->year, $this->previousMonth());
	}

	private function previousYear()
	{
		return $this->year - 1;
	}

	public function previousYearSheet()
	{
		return new CalendarSheet($this->previousYear(), 12);
	}

	public function __toString()
	{
		return "{$this->year}/{$this->month}";
	}

}
