<?php
namespace ZWE\Calendar;

use function ZWE\count_safe;

class CalendarDay
{

	private $tst;
	private $localtime;
	private $year;
	private $month;

	private $user;
	private $availability;
	private $assignments;

	public function __construct($tst, $year, $month, \User $user = null, \Availability $availability = null, $assignments = null)
	{
		$this->tst = $tst;
		$this->localtime = localtime($tst, true);
		$this->year = $year;
		$this->month = $month;
		$this->user = $user;
		$this->availability = $availability;
		$this->assignments = $assignments;
	}

	public function addAvailability()
	{
		if ($this->availability == null) {
			$this->availability = new \Availability();
			$this->availability->User = $this->user;
			$this->availability->datum = strftime('%Y-%m-%d', $this->tst);
			$this->availability->save();
		}
	}

	public function deleteAvailability()
	{
		if ($this->availability != null) {
			$this->availability->delete();
			$this->availability = null;
		}
	}

	public function getTimestamp()
	{
		return $this->tst;
	}

	public function getYear()
	{
		return strftime('%Y', $this->tst);
	}

	public function getMonth()
	{
		return strftime('%m', $this->tst);
	}

	public function getDay()
	{
		return strftime('%d', $this->tst);
	}

	public function getHighestAssignmentState()
	{
		if ($this->assignments != null) {
			$states = array();
			foreach ($this->assignments as $assignment) {
				if ($assignment->state != \Assignment::STATE_DEFAULT) {
					array_push($states, $assignment->state);
				}
			}
			if (in_array(\Assignment::STATE_CANCELLED, $states)) {
				return \Assignment::STATE_CANCELLED;
			} elseif (in_array(\Assignment::STATE_CONFIRMED, $states)) {
				return \Assignment::STATE_CONFIRMED;
			} elseif (in_array(\Assignment::STATE_INVITED, $states)) {
				return \Assignment::STATE_INVITED;
			}
		}
		return \Assignment::STATE_DEFAULT;
	}

	public function hasAssignment()
	{
		return count_safe($this->assignments) > 0;
	}

	function isInCurrentMonth()
	{
		return $this->month == $this->getMonth();
	}

	function isMonday()
	{
		return $this->localtime['tm_wday'] == 1;
	}

	function isSaturday()
	{
		return $this->localtime['tm_wday'] == 6;
	}

	function isAvailable()
	{
		return $this->availability != null;
	}

	function isSunday()
	{
		return $this->localtime['tm_wday'] == 0;
	}

	public function __toString()
	{
		$localtime = localtime($this->tst, true);
		return strftime('%d.%m.%Y %H:%M:%S %z', $this->tst) . " ({$this->tst}) [DST:{$localtime['tm_isdst']}]";
	}

}
