<?php
namespace ZWE\Vergabezeitraum;

use ZWE\Calendar\CalendarDay;
use ZWE\DateStorage\DateStorageException;

class Vergabezeitraum
{

	private static $from_id = 'from';
	private static $to_id = 'to';
	private static $group_vzr = 'VZR';
	private static $group_ewd = 'EWD';

	private $from;
	private $to;

	private $from_localtime;
	private $to_localtime;

	private $min_year;
	private $min_month;
	private $min_day;
	private $max_year;
	private $max_month;
	private $max_day;

	private $from_string;
	private $to_string;

	private $dateStorage;

	public function __construct()
	{
		$this->dateStorage = new \ZWE\DateStorage\DateStorage();
		$this->to = $this->getToPersistent()->format('U');
		$this->from = $this->getFromPersistent()->format('U');

		// wenn der vergabezeitraum aktiv ist, beginn aktualisieren, wenn er in der vergangenheit liegt
		if($this->isActive()) {
			if ($this->from < time()) {
				$newtime = localtime(time(), true);
				$this->from = mktime(0, 0, 0, $newtime['tm_mon'] + 1, $newtime['tm_mday'], $newtime['tm_year'] + 1900);
			}
		}

		$this->from_localtime = localtime($this->from, true);
		$this->to_localtime = localtime($this->to, true);

		$this->min_year = $this->from_localtime['tm_year'] + 1900;
		$this->max_year = $this->to_localtime['tm_year'] + 1900;
		$this->min_month = $this->from_localtime['tm_mon'] + 1;
		$this->max_month = $this->to_localtime['tm_mon'] + 1;
		$this->min_day = $this->from_localtime['tm_mday'];
		$this->max_day = $this->to_localtime['tm_mday'];
		$this->from_string = strftime('%d.%m.%Y', $this->from);
		$this->to_string = strftime('%d.%m.%Y', $this->to);
	}

	public function getStartYear()
	{
		return $this->min_year;
	}

	public function getStartMonth()
	{
		return $this->min_month;
	}

	/**
	 * Gibt zur&uuml;ck, ob der Vergabezeitraum zum aktuellen Zeitpunkt aktiv ist.
	 *
	 * @return bool
	 */
	public function isActive()
	{
		return $this->isActiveOn(new \DateTime());
	}

	/**
	 * Gibt zur&uuml;ck, ob der Vergabezeitraum zum angegebenen Zeitpunkt aktiv ist.
	 *
	 * @param \DateTime $dateTime Zeitpunkt
	 * @return bool
	 */
	public function isActiveOn(\DateTime $dateTime)
	{
		$myDateTime = clone($dateTime);
		$myDateTime->setTime(0, 0, 0);
		$tst = $myDateTime->format("U");
		return $tst >= $this->from && $tst <= $this->to;
	}

	private function isDayIncluded($year, $month, $day)
	{
		return $this->isGEFromDay($year, $month, $day) && $this->isLEToDay($year, $month, $day);
	}

	private function isGEFromMonth($year, $month)
	{
		return ($this->mktime($year, $month, 1) >= $this->mktime($this->min_year, $this->min_month, 1)) ? true : false;
	}

	private function isGEFromDay($year, $month, $day)
	{
		return ($this->mktime($year, $month, $day) >= $this->mktime($this->min_year, $this->min_month, $this->min_day)) ? true : false;
	}

	private function isLEToMonth($year, $month)
	{
		return ($this->mktime($year, $month, 1) <= $this->mktime($this->max_year, $this->max_month, 1)) ? true : false;
	}

	private function isLEToDay($year, $month, $day)
	{
		return ($this->mktime($year, $month, $day) <= $this->mktime($this->max_year, $this->max_month, $this->max_day)) ? true : false;
	}

	public function isMonthIncluded($year, $month)
	{
		return $this->isGEFromMonth($year, $month) && $this->isLEToMonth($year, $month);
	}

	public function isSelectable(CalendarDay $cd)
	{
		if ($this->checkExtraWorkday($cd->getYear(), $cd->getMonth(), $cd->getDay())) {
			return $this->isDayIncluded($cd->getYear(), $cd->getMonth(), $cd->getDay());
		}
		if ($this->isDayIncluded($cd->getYear(), $cd->getMonth(), $cd->getDay())) {
			return $cd->isSaturday() || $cd->isSunday();
		}
		return false;
	}

	public function isYearIncluded($year)
	{
		return ($this->min_year <= $year && $this->max_year >= $year) ? true : false;
	}

	public function checkExtraWorkday($year, $month, $day)
	{
		foreach ($this->getExtraWorkdaysPersistent() as $extra) {
			$y = $extra->getValue()->format('Y');
			$m = $extra->getValue()->format('m');
			$d = $extra->getValue()->format('d');
			if ($y == $year && $m == $month && $d == $day) {
				return true;
			}
		}
		return false;
	}

	public function getFromAsString()
	{
		return $this->from_string;
	}

	public function getToAsString()
	{
		return $this->to_string;
	}

	private function mktime($year, $month, $day)
	{
		return mktime(0, 0, 0, $month, $day, $year);
	}

	public function getFromPersistent()
	{
		try {
			return $this->dateStorage->loadPersistentDate(self::$from_id)->getValue();
		} catch (DateStorageException $ds) {
			return new \DateTime(strftime('%Y-%m-%d', mktime(0, 0, 0, 9, 1, 2009)));
		}
	}

	public function setFromPersistent(\DateTime $dateTime)
	{
		$this->addOrUpdatePersistent(self::$from_id, self::$group_vzr, $dateTime);
	}

	public function getToPersistent()
	{
		try {
			return $this->dateStorage->loadPersistentDate(self::$to_id)->getValue();
		} catch (DateStorageException $ds) {
			return new \DateTime(strftime('%Y-%m-%d', mktime(23, 59, 59, 6, 30, 2010)));
		}
	}

	public function setToPersistent(\DateTime $dateTime)
	{
		$this->addOrUpdatePersistent(self::$to_id, self::$group_vzr, $dateTime);
	}

	private function addOrUpdatePersistent($id, $group, \DateTime $dateTime)
	{
		try {
			$this->dateStorage->addNewPersistentDate($dateTime, $group, $id);
		} catch (\ZWE\DateStorage\DateAlreadyExistsException $ds) {
			$pd = $this->dateStorage->loadPersistentDate($id);
			$pd->setValue($dateTime);
			$pd->save();
		}
	}

	public function addExtraWorkday(\DateTime $dateTime)
	{
		if (!$this->checkExtraWorkdayAlreadyPersistent($dateTime)) {
			$this->addOrUpdatePersistent(null, self::$group_ewd, $dateTime);
		}
	}

	public function checkExtraWorkdayAlreadyPersistent(\DateTime $dateTime)
	{
		$extras = $this->getExtraWorkdaysPersistent();
		foreach ($extras as $e) {
			if ($e->getValue()->format('U') == $dateTime->format('U')) {
				return true;
			}
		}
		return false;
	}

	public function deletePersistent($id)
	{
		return $this->dateStorage->deletePersistentDate($id);
	}

	public function deleteExpiredExtraWorkdays()
	{
		$this->dateStorage->deleteExpiredDatesInGroup(self::$group_ewd);
	}

	public function getExtraWorkdaysPersistent()
	{
		return $this->dateStorage->loadPersistentDateGroup(self::$group_ewd);
	}

}