<?php
namespace ZWE\User;

use ZWE\Container\ZWEApplicationContainer;
use ZWE\UUID;

class PasswordResetService
{

	/**
	 * @var ZWEApplicationContainer
	 */
	private $container;

	public function __construct(ZWEApplicationContainer $container)
	{
		$this->container = $container;
	}

	/**
	 * Findet ein Benutzerkonto mit der angegebenen ID und dem angegebenen Reset-Token.
	 *
	 * @param $id Benutzer-ID
	 * @param $resetToken Reset-Token
	 * @return \User|null
	 * @throws \InvalidArgumentException
	 */
	public function findUserByIdAndResetToken($id, $resetToken)
	{
		if ($id == null) {
			throw new \InvalidArgumentException("User-ID erforderlich!");
		}
		if ($resetToken == null) {
			throw new \InvalidArgumentException("Reset-Token erforderlich!");
		}

		return $this->container->getUserRepo()->findByIdAndPasswordchangeCode($id, $resetToken);
	}

	/**
	 * Passwort-Reset f&uuml;r den Benutzer mit dem angegebenen Benutzernamen
	 * und der angegebenen E-Mailadresse durch beantragen.
	 *
	 * @param $username Benutzername
	 * @param $email E-Mailadresse
	 * @return bool
	 * @throws \InvalidArgumentException
	 */
	public function requestPasswordReset($username, $email)
	{
		if ($username == null) {
			throw new \InvalidArgumentException("Benutzername erforderlich!");
		}
		if ($email == null) {
			throw new \InvalidArgumentException("E-Mailadresse erforderlich!");
		}

		$user = $this->container->getUserRepo()->findByLoginAndEMail($username, $email);
		if ($user != null) {
			$user->chpw_code = UUID::randomUUID();
			$user->save();
			$job = $this->container->getJobFactory()->createSendPasswordResetMailJob()->setUser($user);
			$this->container->getQueue()->queue($job);
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Passwort-Reset durchf&uuml;hren.
	 * @param $id Benutzer-ID
	 * @param $resetToken Reset-Token
	 * @param $newPassword neues Passwort
	 * @throws \InvalidArgumentException
	 */
	public function resetPassword($id, $resetToken, $newPassword)
	{
		if ($id == null) {
			throw new \InvalidArgumentException("User-ID erforderlich!");
		}
		if ($resetToken == null) {
			throw new \InvalidArgumentException("Reset-Token erforderlich!");
		}
		if ($newPassword == null) {
			throw new \InvalidArgumentException("Neues Passwort erforderlich!");
		}

		$user = $this->container->getUserRepo()->find($id);
		if ($user->chpw_code != $resetToken) {
			throw new \InvalidArgumentException("Reset-Token stimmt nicht überein!");
		}

		$user->setPassword($newPassword);
		$user->chpw_code = null;
		$user->save();
	}
}