<?php
namespace ZWE\Repository;

class LicenseRepository extends AbstractRepository
{

	public function find($id)
	{
		return $this->findInternal('License', $id);
	}

	/**
	 * Gibt alle normalen und effektiven Lizenzen für Einzelwettbewerbe als Array für die Verwendung in Dropdown-Listen zurück.
	 * @return array (id => name)
	 */
	public function getAllForDropdown()
	{
		return $this->toSingleDimension($this->getAll(), 'id', 'name');
	}

	public function getAllEinzelReverseOrderForDropdown($onlyEnabled = TRUE)
	{
		return $this->toSingleDimension($this->getAllLicensesInternal(\License::TYPE_EINZEL, $onlyEnabled, \Doctrine::HYDRATE_ARRAY, null, 'l.prio DESC')->execute(), 'id', 'name');
	}

	/**
	 * Gibt alle normalen und effektiven Lizenzen für Einzelwettbewerbe als Doctrine_Collection von License-Objekten zurück.
	 * @return Doctrine_Collection (License)
	 */
	public function getAll()
	{
		return $this->getAllLicensesInternal(\License::TYPE_EINZEL, false, \Doctrine::HYDRATE_RECORD, null, 'l.prio')->execute();
	}

	/**
	 * Gibt alle normalen und effektiven Lizenzen für alle Wettbewerbsarten als Doctrine_Collection von License-Objekten zurück.
	 * @return Doctrine_Collection (License)
	 */
	public function getAllAsObject_EF()
	{
		return $this->getAllLicensesInternal(null, false, \Doctrine::HYDRATE_RECORD, null, 'l.prio')->execute();
	}

	/**
	 * Lädt alle normalen Lizenzstufen für Einzelwettbewerbe.
	 * @param boolean $onlyEnabled nur aktive Lizenzstufen laden
	 * @return Doctrine_Collection von License-Objekten
	 */
	public function getAllLicenses($onlyEnabled = TRUE)
	{
		return $this->toSingleDimension($this->getAllLicensesInternal(\License::TYPE_EINZEL, $onlyEnabled, \Doctrine::HYDRATE_ARRAY)->execute(), 'id', 'name');
	}

	/**
	 * Lädt alle effektiven Lizenzstufen für Einzelwettbewerbe.
	 * @param boolean $onlyEnabled nur aktive Lizenzstufen laden
	 * @return Doctrine_Collection von License-Objekten
	 */
	public function getAllEffectiveLicenses($onlyEnabled = TRUE)
	{
		return $this->toSingleDimension($this->getAllLicensesInternal(\License::TYPE_EINZEL, $onlyEnabled, \Doctrine::HYDRATE_ARRAY, TRUE)->execute(), 'id', 'shortname');
	}

	/**
	 * Interne Laderoutine für Lizenzobjekte.
	 * @param string $type Lizenztyp E, F oder null (beide Typen)
	 * @param boolean $onlyEnabled nur aktivierte Lizenzen laden
	 * @param int $hydration_mode Doctrine Hydration Mode
	 * @param boolean $effective normale oder effektive Lizenzen laden, bei null werden beide Typen geladen
	 * @param string $sort Sortierfeld
	 * @return Doctrine_Query
	 */
	private function getAllLicensesInternal($type, $onlyEnabled, $hydration_mode, $effective = FALSE, $sort = 'l.sortorder')
	{
		$query = \Doctrine_Query::create()
			->select("l.id, l.name, l.shortname")
			->from('License l')
			->orderBy('l.type,' . $sort)
			->setHydrationMode($hydration_mode);
		if ($type !== null) {
			$query->addWhere('l.type = ?', $type);
		}
		if ($onlyEnabled) {
			$query->addWhere('l.enabled = ?', 1);
		}
		if ($effective !== null) { // ATTENTION: do not compare with != null because (false != null) is false; see #74
			if ($effective) {
				$query->addWhere('l.effective = ?', 1);
			} else {
				$query->addWhere('l.effective = ?', 0);
			}
		}
		return $query;
	}

	public function getAllLicenseObjects($enabled, $effective, $types)
	{
		$query = \Doctrine_Query::create()
			->from('License l')
			->orderBy('l.type', 'l.sortorder');
		if ($enabled !== null) {
			$query->addWhere('l.enabled = ?', $enabled ? 1 : 0);
		}
		if ($effective !== null) {
			$query->addWhere('l.effective = ?', $effective ? 1 : 0);
		}
		if ($types !== null) {
			if (is_array($types)) {
				$query->andWhereIn('l.type', $types);
			} else {
				$query->andWhere('l.type = ?', $types);
			}
		}

		return $query->execute();
	}
}
