<?php
namespace ZWE\Repository;

use function ZWE\count_safe;

class EventRepository extends AbstractRepository
{

	const FIELD_CLUB_ID = 'club_id';
	const FIELD_LOCATION_ID = 'location_id';
	const FIELD_NAME = 'name';

	public function find($id)
	{
		return $this->findInternal('Event', $id);
	}

	/**
	 * Findet alle Veranstaltungen im gegebenen Zeitraum
	 * @param DateTime $from Beginn des Zeitraums (inkl.)
	 * @param DateTime $to Ende des Zeitraums (inkl.)
	 * @param bool $onlyWithTeams nur Veranstaltungen mit WR-Teams suchen (default = false)
	 * @param bool $onlyWithCompetitions nur Veranstaltungen mit WR-Teams und Turnieren suchen (nur wirksam bei $onlyWithTeams = true, default = false)
	 * @param array|Doctrine_Collection $mandanten Besitzer
	 * @return \Doctrine_Collection
	 */
	public function findBetween(\DateTime $from, \DateTime $to, $onlyWithTeams = false, $onlyWithCompetitions = false, $mandanten = null)
	{
		$query = \Doctrine_Query::create()
			->select('e.*, t.*, c.*, cl.*, l.*')
			->from('Event e')
			->innerJoin('e.Club cl')
			->innerJoin('e.Location l')
			->leftJoin('e.WRTeam t')
			->leftJoin('t.Competitions c')
			->where('e.datum >= ?', $from->format('Y-m-d'))
			->andWhere('e.datum <= ?', $to->format('Y-m-d'))
			->orderBy('e.datum, e.name, e.id');
		if ($onlyWithTeams) {
			$query->addWhere('t.event_id IS NOT NULL');
			if ($onlyWithCompetitions) {
				$query->addWhere('c.team_id IS NOT NULL');
			}
		}
		if ($mandanten != null) {
			$mandanten_ids = \ZWE\Mandator\MandatorUtils::asIdArray($mandanten);
			if (count_safe($mandanten_ids) > 0) {
				$query->andWhereIn('e.mandator_id', $mandanten_ids);
			}
		}
		return $query->execute();
	}

	public function getAuditPager($page = 1, $results = 5, $owners = null)
	{
		return new \Doctrine_Pager($this->getAuditQuery($owners), $page, $results);
	}

	private function getAuditQuery($owners = null)
	{
		$query = \Doctrine_Query::create()
			->select('e.*, COUNT(t.id) AS team_count')
			->addSelect('(SELECT COUNT(a2.id) FROM Assignmentplan a2 WHERE a2.event_id = e.id AND a2.valid = 1) valid_assignmentplan_count')
			->from('Event e')
			->leftJoin('e.WRTeam t')
			->leftJoin('e.Assignmentplans a')
			->where('e.datum >= NOW()')
			->having('team_count = 0 OR valid_assignmentplan_count = 0')
			->groupBy('e.id')
			->orderBy('e.datum, e.name, e.id');
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			$query->andWhereIn('e.mandator_id', $owner_ids);
		}
		return $query;
	}

	public function pagedQuery($page = 1, $maxPerPage = 10, $inFutureOnly = true, $filter = array(), $owners = null)
	{
		return $this->pagedQueryInternal($page, $maxPerPage, $inFutureOnly, $filter, $owners)->execute();
	}

	private function pagedQueryInternal($page = 1, $maxPerPage = 10, $inFutureOnly = true, $filter = array(), $owners = null)
	{
		$query = \Doctrine_Query::create()
			->from('Event e')
			->innerJoin('e.Club c')
			->innerJoin('e.Location l')
			->leftJoin('e.Assignmentplans a')
			->leftJoin('e.WRTeam t')
			->orderBy('e.datum, e.name, e.id');
		if ($inFutureOnly) {
			$query->andWhere('e.datum >= ?', date('Y-m-d'));
		}
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			$query->andWhereIn('e.mandator_id', $owner_ids);
		}
		$this->augmentDynamicQuery($query, $filter);
		$pager = new \Doctrine_Pager($query, $page, $maxPerPage);
		return $pager;
	}

	public function pagedQueryWithPagerResult($page = 1, $maxPerPage = 10, $inFutureOnly = true, $filter = array(), $owners = null)
	{
		return $this->pagedQueryInternal($page, $maxPerPage, $inFutureOnly, $filter, $owners);
	}

}