<?php
namespace ZWE\Repository;

use function ZWE\count_safe;

class ClubRepository extends AbstractRepository
{

	public function allNames($params = null)
	{
		$ret = array();
		$clubs = $this->allByParams($params);
		foreach ($clubs as $club) {
			if (!isset($ret[$club->LTV->name])) {
				$ret[$club->LTV->name] = array();
			}
			$ret[$club->LTV->name][(string)$club->id] = $club->name;
		}
		return $ret;
	}

	public function allNamesAsArrayGroupedByLTV($params = null, $owners = null, $withMandator = false)
	{
		$ret = array();
		$clubs = $this->allByParams($params, $owners);
		foreach ($clubs as $club) {
			if (!isset($ret[$club->LTV->name])) {
				$ret[$club->LTV->name] = array();
			}
			$ret[$club->LTV->name][(string)$club->id] = $club->name;
			if ($withMandator) {
				$ret[$club->LTV->name][(string)$club->id] .= ' (' . $club->Owner->name . ')';
			}
		}
		return $ret;
	}

	public function allNamesExternalAsArrayGroupedByCountry()
	{
		$ret = array();
		$clubs = $this->allByParams(['ltv_id' => 'EXT', 'has_wr' => true], null);
		foreach ($clubs as $club) {
			if (!isset($ret[$club->Country->name])) {
				$ret[$club->Country->name] = array();
			}
			$ret[$club->Country->name][(string)$club->id] = $club->name;
		}
		return $ret;
	}

	private function allByParams($params = null, $owners = null)
	{
		$query = \Doctrine_Query::create()
			->select('c.*, m.*, l.*')
			->from('Club c')
			->innerJoin('c.Owner m')
			->innerJoin('c.LTV l')
			->orderBy('l.name, c.name');
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			if (count_safe($owner_ids) > 0) {
				$query->andWhereIn('c.mandator_id', $owner_ids);
			}
		}
		if ($params != null && is_array($params)) {
			foreach ($params as $name => $value) {
				$query->addWhere("c.{$name} = ?", $value);
			}
		}
		return $query->execute();
	}

	public function find($id)
	{
		return $this->findInternal('Club', $id);
	}

	public function findByDTVVnum($vnum)
	{
		$query = \Doctrine_Query::create()
			->select('c.*')
			->from('Club c')
			->where('c.vnum_dtv = ?', $vnum);
		return $query->fetchOne();
	}

	public function findAll() {
		return $this->allByParams();
	}

	/**
	 * @param int $page
	 * @param int $maxPerPage
	 * @param string $ltv
	 * @param string $sortname
	 * @param string $sortorder
	 * @param string $queryfield
	 * @param string $queryvalue
	 * @param null $owners
	 * @return \Doctrine_Collection
	 */
	public function pagedQuery($page = 1, $maxPerPage = 10, $ltv = '%', $sortname = 'name', $sortorder = 'asc', $queryfield = 'name', $queryvalue = '%', $owners = null)
	{
		$pager = $this->pagedQueryWithPagerResult($page, $maxPerPage, $ltv, $sortname, $sortorder, $queryfield, $queryvalue, $owners);
		return $pager->execute();
	}

	/**
	 * @param int $page
	 * @param int $maxPerPage
	 * @param string $ltv
	 * @param string $sortname
	 * @param string $sortorder
	 * @param string $queryfield
	 * @param string $queryvalue
	 * @param null $owners
	 * @return \Doctrine_Pager
	 */
	public function pagedQueryWithPagerResult($page = 1, $maxPerPage = 10, $ltv = '%', $sortname = 'name', $sortorder = 'asc', $queryfield = 'name', $queryvalue = '%', $owners = null)
	{
		$query = \Doctrine_Query::create()
			->from('Club c')
			->leftJoin('c.Users u')
			->where("c.{$queryfield} LIKE ?", $queryvalue)
			->andWhere('c.ltv_id LIKE ?', $ltv)
			->orderBy("c.{$sortname} {$sortorder}");
		if ($owners != null) {
			$owner_ids = \ZWE\Mandator\MandatorUtils::asIdArray($owners);
			if (count_safe($owner_ids) > 0) {
				$query->andWhereIn('c.mandator_id', $owner_ids);
			}
		}
		return new \Doctrine_Pager($query, $page, $maxPerPage);
	}
}