<?php

namespace ZWE\Queue;

use Psr\Log\LoggerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use ZWE\Container\LoggingService;
use function ZWE\count_safe;

class DatabaseQueueDriver implements QueueDriver
{

	/**
	 * @var ValidatorInterface
	 */
	private $validator;

	/**
	 * @var LoggerInterface
	 */
	private $logger;

	public function __construct(ValidatorInterface $validator, LoggingService $loggingService)
	{
		$this->validator = $validator;
		$this->logger = $loggingService->createLogger(DatabaseQueueDriver::class);
	}

	public function queue(Job $job)
	{
		$violations = $this->validator->validate($job);
		if ($violations->count() > 0) {
			$this->logger->error('job is invalid', ['violations' => $violations]);
			throw new \InvalidArgumentException('Job is invalid!');
		}

		$queuedJob = new \QueuedJob();
		$queuedJob->class = get_class($job);
		$queuedJob->payload = serialize($job);
		$queuedJob->setDateTimeObject('queued', new \DateTime());
		$queuedJob->save();

		$this->logger->info('job queued', ['id' => $queuedJob->id, 'class' => $queuedJob->class]);
		$this->logger->debug('dumping job data', ['data' => $queuedJob->payload]);
	}

	/**
	 * @return \QueuedJob|null
	 */
	public function getNextJob()
	{
		$results = \Doctrine_Query::create()
			->from(\QueuedJob::class)
			->whereIn('status', [\QueuedJob::JOB_QUEUED, \QueuedJob::JOB_ERROR])
			->orderBy('queued ASC')
			->limit(1)
			->execute();

		if ($results && $results->count() > 0) {
			return $results[0];
		} else {
			return null;
		}
	}

	public function pageJobs($page, $sort = 'id', $order = 'DESC', $size = 10)
	{
		$count = \Doctrine_Query::create()
			->from(\QueuedJob::class)
			->count();

		$results = \Doctrine_Query::create()
			->from(\QueuedJob::class)
			->orderBy("{$sort} ${order}")
			->limit($size)
			->offset(($page - 1) * $size)
			->execute([], \Doctrine::HYDRATE_ARRAY);

		if ($results && count_safe($results) > 0) {
			return (object)[
				'data' => $results,
				'page' => $page,
				'pages' => ceil($count / $size),
			];
		} else {
			return (object)[
				'pages' => 0,
				'page' => 0,
				'data' => []
			];
		}
	}

	public function requeue($id)
	{
		if (!$id) {
			throw new \InvalidArgumentException('ID required');
		}

		$entry = \Doctrine_Query::create()->from(\QueuedJob::class)
			->where('id = ?', $id)
			->limit(1)
			->execute();
		if ($entry && $entry->count() > 0) {
			$entry[0]->status = \QueuedJob::JOB_QUEUED;
			$entry[0]->tries = 0;
			$entry[0]->log = null;
			$entry[0]->started = null;
			$entry[0]->finished = null;

			$entry[0]->save();
		} else {
			throw new \RuntimeException("Job #{$id} existiert nicht!");
		}
	}
}
