<?php
namespace ZWE\Mandator;
use ZWE\User\UserContext;

/**
 * Statische Helferklasse, mit der der Zugriff auf Mandanten überprüft oder
 * verifiziert werden kann.
 *
 * Die Funktionen überprüfen lediglich Objekte, die tatsächlich mandantenbasiert
 * sind. Andere Objekte können zwar als Parameter übergeben werden, eine
 * Prüfung findet in diesen Fällen jedoch nicht statt.
 *
 * Die verifyXYZ-Funktionen führen bei nicht ausreichender Berechtigung zu
 * einer InsufficientPrivilegeException, wohingegen die checkXYZ-Funktionen
 * lediglich überprüfen und ein boolsches Ergebnis zurückliefern. Intern
 * verwenden die checkXYZ-Funktionen die entsprechende verifyXYZ-Funktion,
 * fangen aber die InsufficientPrivilegeException ab und liefern den
 * entsprechenden Rückgabewert.
 *
 * @author  Hendrik Heneke <hendrik@pixelwunder.net>
 * @since   2.5
 */
class MandatorPrivilegeEscalator
{

	/**
	 * Überprüft, ob der aktuelle Benutzer das Verwaltungsrecht für das
	 * mandantenbasiert Objekt <code>$object</code> besitzt. Ohne Angabe des
	 * Parameters <code>$write</code> wird der Lesezugriff überprüft. Objekte,
	 * die nicht mandantenbasiert sind oder auf die sich das Verwaltungsrecht
	 * nicht bezieht (Event) können als Parameter übergeben werden, allerdings
	 * findet dann keine Überprüfung statt.
	 *
	 * @param mixed $object mandantenbasierte Objekt (Club, User, Location)
	 * @param bool $write schreibenden Zugriff überprüfen (default false)
	 * @return bool
	 */
	public static function checkAdminister($object, $write = false)
	{
		try {
			self::verifyAdminister($object, $write);
			return true;
		} catch (InsufficientPrivilegeException $ipe) {
			return false;
		}
	}

	/**
	 * Überprüft, ob der aktuelle Benutzer das Verwaltungsrecht für den
	 * aktuellen Mandanten besitzt. Ohne Angabe des Parameters
	 * <code>$write</code> wird der Lesezugriff überprüft.
	 *
	 * @param bool $write schreibenden Zugriff überprüfen (default false)
	 * @return bool
	 */
	public static function checkAdministerCurrentMandator($write = false)
	{
		try {
			self::verifyAdministerCurrentMandator($write);
			return true;
		} catch (InsufficientPrivilegeException $ipe) {
			return false;
		}
	}

	/**
	 * Überprüft, ob der aktuelle Benutzer das Planungsrecht für die
	 * Veranstaltung <code>$event</code> besitzt. Ohne Angabe des
	 * Parameters <code>$write</code> wird der Lesezugriff überprüft.
	 *
	 * @param Event $event Veranstaltung
	 * @param bool $write schreibenden Zugriff überprüfen (default false)
	 * @return bool
	 */
	public static function checkPlan(\Event $event, $write = false)
	{
		try {
			self::verifyPlan($event, $write);
			return true;
		} catch (InsufficientPrivilegeException $ipe) {
			return false;
		}
	}

	/**
	 * Überprüft, ob der aktuelle Benutzer das Planungsrecht für die
	 * den aktuellen Mandanten besitzt. Ohne Angabe des
	 * Parameters <code>$write</code> wird der Lesezugriff überprüft.
	 *
	 * @param bool $write schreibenden Zugriff überprüfen (default false)
	 * @return bool
	 */
	public static function checkPlanCurrentMandator($write = false)
	{
		try {
			self::verifyPlanCurrentMandator($write);
			return true;
		} catch (InsufficientPrivilegeException $ipe) {
			return false;
		}
	}

	/**
	 * Verifiziert Verwaltungsrecht auf das Objekt <code>$object</code>.
	 * Ohne Angabe des Parameters <code>$write</code> wird der Lesezugriff
	 * verifiziert. Objekte, die nicht mandantenbasiert sind oder auf
	 * die sich das Verwaltungsrecht nicht bezieht (Event) können als
	 * Parameter übergeben werden, allerdings findet dann keine Überprüfung
	 * statt.
	 *
	 * @param mixed $object mandantenbasierte Objekt (Club, User, Location)
	 * @param bool $write schreibenden Zugriff verifizieren (default false)
	 *
	 * @throws InsufficientPrivilegeException falls der Benutzer nicht
	 * berechtigt ist, die Operation auszuführen
	 */
	public static function verifyAdminister($object, $write = false)
	{
		$user = UserContext::getInstance()->getUser();
		switch (get_class($object)) {
			case 'Club':
			case 'Location':
				if (!$write) {
					self::verifyMandatorGrant($object->Owner, $user,
						true, null, null, null);
				} else {
					self::verifyMandatorGrant($object->Owner, $user,
						null, null, true, null);
				}
				break;
			case 'User':
				if ($object->isWR()) {
					if (!$write) {
						self::verifyMandatorGrant($object->Club->Owner, $user,
							true, null, null, null);
					} else {
						self::verifyMandatorGrant($object->Club->Owner, $user,
							null, null, true, null);
					}
				} else {
					throw new InsufficientPrivilegeException('Zugriff nicht
                        möglich, da ZWE- oder Adminbenutzer nicht
                        mandantenbasiert sind!');
				}
				break;
			default:
				break;
		}
	}

	/**
	 * Verifiziert das Verwaltungsrecht für den aktuellen Mandanten. Ohne
	 * Angabe des Parameters <code>$write</code> wird der Lesezugriff
	 * verifiziert.
	 *
	 * @param bool $write schreibenden Zugriff verifizieren (default false)
	 *
	 * @throws InsufficientPrivilegeException falls der Benutzer nicht
	 * berechtigt ist, die Operation auszuführen
	 */
	public static function verifyAdministerCurrentMandator($write = false)
	{
		$user = UserContext::getInstance()->getUser();
		if (!$write) {
			self::verifyMandatorGrant(MandatorContext::getInstance()->
				getMandator(), $user, true, null, null, null);
		} else {
			self::verifyMandatorGrant(MandatorContext::getInstance()->
				getMandator(), $user, null, null, true, null);
		}
	}

	/**
	 * Verifiziert Planungsrecht auf die Veranstaltung <code>$event</code>.
	 * Ohne Angabe des Parameters <code>$write</code> wird der Lesezugriff
	 * verifiziert.
	 *
	 * @param Event $event Veranstaltung
	 * @param bool $write schreibenden Zugriff verifizieren (default false)
	 *
	 * @throws InsufficientPrivilegeException falls der Benutzer nicht
	 * berechtigt ist, die Operation auszuführen
	 */
	public static function verifyPlan(\Event $event, $write = false)
	{
		$user = UserContext::getInstance()->getUser();
		if (!$write) {
			self::verifyMandatorGrant($event->Owner, $user,
				null, true, null, null);
		} else {
			self::verifyMandatorGrant($event->Owner, $user,
				null, null, null, true);
		}
	}

	/**
	 * Verifiziert das Planungsrecht für den aktuellen Mandanten. Ohne
	 * Angabe des Parameters <code>$write</code> wird der Lesezugriff
	 * verifiziert.
	 *
	 * @param bool $write schreibenden Zugriff verifizieren (default false)
	 *
	 * @throws InsufficientPrivilegeException falls der Benutzer nicht
	 * berechtigt ist, die Operation auszuführen
	 */
	public static function verifyPlanCurrentMandator($write = false)
	{
		$user = UserContext::getInstance()->getUser();
		if (!$write) {
			self::verifyMandatorGrant(MandatorContext::getInstance()->
				getMandator(), $user, null, true, null, null);
		} else {
			self::verifyMandatorGrant(MandatorContext::getInstance()->
				getMandator(), $user, null, null, null, true);
		}
	}

	/**
	 * Interne Prüfroutine für den Zugriff auf Mandanten.
	 *
	 * @param Mandator $owner
	 * @param User $user
	 * @param bool $admin_read
	 * @param bool $plan_read
	 * @param bool $admin_write
	 * @param bool $plan_write
	 *
	 * @throws InsufficientPrivilegeException
	 */
	private static function verifyMandatorGrant($owner, $user,
	                                            $admin_read, $plan_read, $admin_write, $plan_write)
	{
		if ($owner == null) {
			throw new InsufficientPrivilegeException(
				'Zugriff nicht möglich, da kein Mandant ausgewählt ist!');
		}
		if ($user == null) {
			throw new InsufficientPrivilegeException(
				'Zugriff nicht möglich, da kein authentisierter
				ZWE-Benutzer angemeldet ist!');
		} else {
			if (!$owner->isOperatedByUserWithACL($user,
				$admin_read, $plan_read, $admin_write, $plan_write)
			) {
				throw new InsufficientPrivilegeException('Zugriff nicht möglich,
                   da Sie nicht die erforderlichen Berechtigungen für den
                   Mandanten "' . $owner->name . '" besitzen!');
			}
		}
	}

}