<?php
namespace ZWE\Mail;

use ZWE\Container\ZWEApplicationContainer;
use ZWE\FileStorage\File;
use ZWE\Mail\Template\Template;
use ZWE\Settings\SettingsRegistry;
use function ZWE\count_safe;

class MessageCreator
{

	const CONTENT_TYPE_HTML = 'text/html';

	/**
	 * @var ZWEApplicationContainer
	 */
	private $app;

	/**
	 * @var bool
	 */
	private $loadFunctions;

	public function __construct(ZWEApplicationContainer $app, $loadFunctions = false)
	{
		$this->app = $app;
		$this->loadFunctions = $loadFunctions;
	}

	/**
	 * @param \User $user
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createPasswordResetMessage(\User $user)
	{
		$variables = $this->getMessageVariablesForUser($user);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject('Zurücksetzen Ihres Passworts'))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($user));

		$template = $this->createTemplate('passwortvergessen');
		$template->render($this->toMessageContext($variables, array('user' => $user)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \User $user
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createUserActivatedMessage(\User $user)
	{
		$variables = $this->getMessageVariablesForUser($user);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject('Neues Benutzerkonto aktiviert'))
			->setFrom($variables->getSenderArray())
			->setTo($variables->getRegistrationRecipient());

		$template = $this->createTemplate('zwe_benutzer_aktiviert');
		$template->render($this->toMessageContext($variables, array('user' => $user)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \User $user
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createUserLoginChangedMessage(\User $user)
	{
		$variables = $this->getMessageVariablesForUser($user);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject('Änderung Ihres Benutzernamens'))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($user))
			->setBcc($variables->getZWERecipient());

		$template = $this->createTemplate('wr_benutzername');
		$template->render($this->toMessageContext($variables, ['user' => $user]));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \User $user
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createUserValidationMessage(\User $user)
	{
		$variables = $this->getMessageVariablesForUser($user);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject('Aktivierung Ihres Benutzerkontos'))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($user));

		$template = $this->createTemplate('wr_activate');
		$template->render($this->toMessageContext($variables, array('user' => $user)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \User $user
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createUserUnlockedMessage(\User $user)
	{
		$variables = $this->getMessageVariablesForUser($user);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject('Freischaltung Ihres Benutzerkontos'))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($user));

		$template = $this->createTemplate('wr_freigeschaltet');
		$template->render($this->toMessageContext($variables, array('user' => $user)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 */
	public function createInvitationMesssage(\Assignment $einsatz, $bemerkungen)
	{
		$variables = $this->getMessageVariablesForMandator($einsatz->WRTeam->Event->Owner);

		$subject = null;
		$templateName = null;
		if ($einsatz->User->isAvailable($einsatz->WRTeam->Event->getDateTimeObject('datum'))) {
			// einsatz nach freigabe
			$templateName = 'wr_einsatz';
			$subject = "Wertungsrichtereinsatz am " . $einsatz->WRTeam->Event->getDateTimeObject('datum')->format('d.m.Y');
		} else {
			// einsatz ohne freigabe
			$templateName = 'wr_einladung';
			$subject = "Einladung zum Wertungsrichtereinsatz am " . $einsatz->WRTeam->Event->getDateTimeObject('datum')->format('d.m.Y');
		}

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject($subject))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($einsatz->User));

		$template = $this->createTemplate($templateName);
		$template->render($this->toMessageContext($variables, array('einsatz' => $einsatz, 'anmerkung' => $bemerkungen)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createAssignmentCancelledMessage(\Assignment $einsatz, $bemerkungen)
	{
		$subject = "Ihre Absage zum Wertungsrichtereinsatz am " . $einsatz->WRTeam->Event->getDateTimeObject('datum')->format('d.m.Y');
		return $this->createAssignmentChangedMessage($einsatz, $subject, $bemerkungen);
	}

	/**
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createAssignmentConfirmedMessage(\Assignment $einsatz, $bemerkungen)
	{
		$subject = "Ihre Zusage zum Wertungsrichtereinsatz am " . $einsatz->WRTeam->Event->getDateTimeObject('datum')->format('d.m.Y');
		return $this->createAssignmentChangedMessage($einsatz, $subject, $bemerkungen);
	}

	/**
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createAssignmentDeletedMessage(\Assignment $einsatz, $bemerkungen)
	{
		$subject = "Absage des Wertungsrichtereinsatzes am " . $einsatz->WRTeam->Event->getDateTimeObject('datum')->format('d.m.Y');
		return $this->createAssignmentChangedMessage($einsatz, $subject, $bemerkungen);
	}

	/**
	 * @param \Assignment $einsatz
	 * @param $subject
	 * @param $bemerkungen
	 * @return \Swift_Mime_SimpleMessage
	 */
	private function createAssignmentChangedMessage(\Assignment $einsatz, $subject, $bemerkungen)
	{
		$variables = $this->getMessageVariablesForMandator($einsatz->WRTeam->Event->Owner);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject($subject))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($einsatz->User));

		$template = $this->createTemplate('wr_assignment_changed');
		$template->render($this->toMessageContext($variables, array('einsatz' => $einsatz, 'bemerkungen' => $bemerkungen)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \Assignment $einsatz
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createReminderMessage(\Assignment $einsatz)
	{
		$variables = $this->getMessageVariablesForMandator($einsatz->WRTeam->Event->Owner);

		$subject = "Erinnerung Zu- oder Absage Wertungsrichtereinsatz am " . $einsatz->WRTeam->Event->getDateTimeObject('datum')->format('d.m.Y');

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject($subject))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($einsatz->User));

		$template = $this->createTemplate('wr_erinnerung');
		$template->render($this->toMessageContext($variables, array('einsatz' => $einsatz)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \Event $event
	 * @param \WRTeam $team
	 * @param \Assignment $einsatz
	 * @param $bemerkungen
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createEventCancelledMessage(\Event $event, \WRTeam $team, \Assignment $einsatz, $bemerkungen)
	{
		$variables = $this->getMessageVariablesForMandator($event->Owner);
		$subject = "Absage des Wertungsrichtereinsatzes am " . $event->getDateTimeObject('datum')->format('d.m.Y') . " (" . $einsatz->WRTeam->name . ")";

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject($subject))
			->setFrom($variables->getSenderArray())
			->setTo($this->userRecipientArray($einsatz->User));

		$template = $this->createTemplate('wr_veranstaltungsabsage');
		$template->render($this->toMessageContext($variables, array('einsatz' => $einsatz, 'anmerkungen' => $bemerkungen)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \Assignment $einsatz
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createCancellationMessage(\Assignment $einsatz)
	{
		$variables = $this->getMessageVariablesForMandator($einsatz->WRTeam->Event->Owner);
		$subject = "Absage f. {$einsatz->WRTeam->Event->getDateTimeObject('datum')->format('d.m.Y')} / {$einsatz->WRTeam->Event->Club->name} / {$einsatz->WRTeam->name}";

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject($subject))
			->setFrom($variables->getSenderArray())
			->setTo($variables->getCancellationRecipient());

		$template = $this->createTemplate('zwe_absage');
		$template->render($this->toMessageContext($variables, array('einsatz' => $einsatz)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \ChangeEmailRequest $request
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createValidateNewEmailMessage(\ChangeEmailRequest $request)
	{
		$variables = $this->getMessageVariablesForUser($request->User);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject("Überprüfung Ihrer neuen E-Mailadresse"))
			->setFrom($variables->getSenderArray())
			->setTo($request->email);

		$template = $this->createTemplate('wr_emailaendern');
		$template->render($this->toMessageContext($variables, array('changerequest' => $request)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \Mandator $mandant
	 * @param $subject
	 * @param $mailHtml
	 * @param array $bcc
	 * @param null|string|string[] $attachments
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createCustomMessage(\Mandator $mandant, $subject, $mailHtml, $bcc = array(), $attachments = null)
	{
		$variables = $this->getMessageVariablesForMandator($mandant);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject($subject))
			->setFrom($variables->getSenderArray())
			->setTo($variables->getZWERecipient());

		if ($bcc != null && count_safe($bcc) > 0) {
			$message->setBcc($bcc);
		}

		$files = array();
		if ($attachments != null) {
			if (is_array($attachments)) {
				$files = $attachments;
			} else {
				$files[] = $attachments;
			}
		}
		foreach ($files as $file) {
			if (is_file($file) && is_readable($file)) {
				$message->attach(\Swift_Attachment::fromPath($file));
			}
		}

		$template = $this->createTemplate('custom');
		$template->render($this->toMessageContext($variables, ['mail_html' => $mailHtml]));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param \Event $event
	 * @param File $pdf
	 * @param File $zwex
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createAssignmentplanMessage(\Event $event, File $pdf, File $zwex)
	{
		$variables = $this->getMessageVariablesForMandator($event->Owner);
		$subject = "Einsatzplan für " . $event->name . " am " . $event->getDateTimeObject('datum')->format('d.m.Y');

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject($subject))
			->setFrom($variables->getSenderArray())
			->setTo($event->Club->email)
			->setCc($variables->getPlanRecipient());

		$storage = $this->app->getNewEventAttachmentStorage($event);
		foreach ($storage->listFiles() as $file) {
			$message->attach(\Swift_Attachment::fromPath($file->getAbsolutePath()));
		}

		$template = $this->createTemplate('verein_einsatzplan');
		$template->render($this->toMessageContext($variables, array('event' => $event, 'gdprEnabled' =>
			SettingsRegistry::getInstance()->get('zwe_gdpr_enabled')->getSystemValue())));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		if (is_file($pdf->getAbsolutePath()) && is_readable($pdf->getAbsolutePath())) {
			$message->attach(\Swift_Attachment::fromPath($pdf->getAbsolutePath()));
		}
		if (is_file($zwex->getAbsolutePath()) && is_readable($zwex->getAbsolutePath())) {
			$message->attach(\Swift_Attachment::fromPath($zwex->getAbsolutePath()));
		}

		return $message;
	}

	/**
	 * @param \User $user
	 * @return \Swift_Mime_SimpleMessage
	 */
	public function createUserDeletedMessage(\User $user)
	{
		$variables = $this->getMessageVariablesForUser($user);

		$message = \Swift_Message::newInstance()
			->setSubject($variables->formatSubject('Benutzerkonto gelöscht'))
			->setFrom($variables->getSenderArray())
			->setTo($variables->getRegistrationRecipient());

		$template = $this->createTemplate('zwe_benutzer_geloescht');
		$template->render($this->toMessageContext($variables, array('user' => $user)));

		$message->addPart($template->getContentHtml(), self::CONTENT_TYPE_HTML);

		return $message;
	}

	/**
	 * @param $templateName
	 * @return Template
	 * @throws \InvalidArgumentException
	 */
	private function createTemplate($templateName)
	{
		if ($templateName == null) {
			throw new \InvalidArgumentException("Template-Name erforderlich!");
		}
		$mailText = $this->app->getMailTextService()->find($templateName);
		if($mailText != null) {
			return new Template($templateName, $this->loadFunctions, $mailText->text_html);
		}
		else {
			return new Template($templateName, $this->loadFunctions);
		}
	}

	/**
	 * @param \User $user
	 * @return MessageVariables
	 */
	private function getMessageVariablesForUser(\User $user)
	{
		if ($user->isWR()) {
			return $this->getMessageVariablesForMandator($user->Club->Owner);
		} else {
			return $this->getMessageVariablesDefault();
		}
	}

	/**
	 * @param \Mandator $mandator
	 * @return MessageVariables
	 */
	private function getMessageVariablesForMandator(\Mandator $mandator)
	{
		return new MessageVariables(
			$this->getMandatorSettingValue($mandator, 'mail_subject_prefix'),
			$this->getMandatorSettingValue($mandator, 'mail_from'),
			$this->getMandatorSettingValue($mandator, 'mail_zwe_sender'),
			$this->getMandatorSettingValue($mandator, 'mail_zwe_recipient'),
			$this->getMandatorSettingValue($mandator, 'mail_reg_recipient'),
			$this->getMandatorSettingValue($mandator, 'mail_cancel_recipient'),
			$this->getMandatorSettingValue($mandator, 'mail_plan_recipient'),
			$this->getMandatorSettingValue($mandator, 'mail_footer_html'),
			$this->getSystemSettingValue('zwe_use_addresses'),
			$this->getSystemSettingValue('zwe_use_birthyear')
		);
	}

	/**
	 * @return MessageVariables
	 */
	private function getMessageVariablesDefault()
	{
		return new MessageVariables(
			$this->getSystemSettingValue('mail_subject_prefix'),
			$this->getSystemSettingValue('mail_from'),
			$this->getSystemSettingValue('mail_zwe_sender'),
			$this->getSystemSettingValue('mail_zwe_recipient'),
			$this->getSystemSettingValue('mail_reg_recipient'),
			$this->getSystemSettingValue('mail_cancel_recipient'),
			$this->getSystemSettingValue('mail_plan_recipient'),
			$this->getSystemSettingValue('mail_footer_html'),
			$this->getSystemSettingValue('zwe_use_addresses'),
			$this->getSystemSettingValue('zwe_use_birthyear')
		);
	}


	/**
	 * @param $key
	 * @return mixed
	 */
	private function getSystemSettingValue($key)
	{
		return $this->app->getSettingsService()->get($key)->getValue();
	}

	/**
	 * @param \Mandator $mandator
	 * @param $key
	 * @return mixed
	 */
	private function getMandatorSettingValue(\Mandator $mandator, $key)
	{
		return $this->app->getSettingsService()->get($key)->getMandatorValue($mandator->id);
	}

	/**
	 * @param \User $user
	 * @return array
	 */
	private function userRecipientArray(\User $user)
	{
		return array($user->email => sprintf('%1$s %2$s', $user->name, $user->surname));
	}

	/**
	 * @param MessageVariables $variables
	 * @param array $context
	 * @return array
	 */
	private function toMessageContext(MessageVariables $variables, array $context)
	{
		$base = array(
			'footer_html' => $variables->getFooterHtml(),
			'use_addresses' => $variables->isUseAddresses(),
			'use_birthyear' => $variables->isUseBirthyear(),
		);
		return array_merge($base, $context);
	}
}
