<?php

namespace ZWE\Mail;

use ZWE\Container\ZWEApplicationContainer;
use ZWE\Mail\Template\Template;
use ZWE\Settings\SettingsRegistry;

class MailTextService
{

	/**
	 * @var ZWEApplicationContainer
	 */
	private $app;

	private static $descriptors = array(
		'impressum' => [
			'category' => 'Frontend-Artikel',
			'description' => 'Impressum',
		],
		'datenschutzerklaerung' => [
			'category' => 'Frontend-Artikel',
			'description' => 'angepasste Datenschutzerklärung'
		],
		'passwortvergessen' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'Passworterinnerung'
		],
		'verein_einsatzplan' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'Einsatzplan'
		],
		'wr_activate' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'Aktivierung Benutzerkonto'
		],
		'wr_einladung' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'WR-Einladung (Einsatz ohne Freigabe)'
		],
		'wr_einsatz' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'WR-Einsatz (Einsatz mit Freigabe)'
		],
		'wr_emailaendern' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'Überprüfung neue E-Mailadresse'
		],
		'wr_erinnerung' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'Erinnerung an Zu- oder Absage'
		],
		'wr_freigeschaltet' => [
			'category' => 'E-Mail Zusatztext',
			'description' => 'Freischaltung Benutzerkonto'
		],
	);

	private static $defaultArticles = array(
		'datenschutzerklaerung' => 'article_datenschutzerklaerung'
	);

	public function __construct(ZWEApplicationContainer $app)
	{
		$this->app = $app;
	}

	/**
	 * @param $id
	 * @param $html
	 * @return \EmailText
	 * @throws \InvalidArgumentException
	 */
	public function createOrUpdate($id, $html)
	{
		if ($id == null) {
			throw new \InvalidArgumentException("Artikel-ID erforderlich!");
		}

		$et = $this->app->getEmailTextRepo()->find($id);
		if ($et == null) {
			$et = new \EmailText();
			$et->id = $id;
		}
		$et->text_html = $html;
		$et->save();

		return $et;
	}

	/**
	 * @param $id
	 * @throws \InvalidArgumentException
	 */
	public function delete($id)
	{
		if ($id == null) {
			throw new \InvalidArgumentException("Artikel-ID erforderlich!");
		}

		$et = $this->app->getEmailTextRepo()->find($id);
		if ($et != null) {
			$et->delete();
		}
	}

	/**
	 * @param $id
	 * @return \EmailText|null
	 * @throws \InvalidArgumentException
	 */
	public function find($id)
	{
		if ($id == null) {
			throw new \InvalidArgumentException("Artikel-ID erforderlich!");
		}

		$article = $this->app->getEmailTextRepo()->find($id);
		if (!$article) {
			$article = $this->loadDefaultArticleIfExists($id);
		}
		return $article;
	}

	private function loadDefaultArticleIfExists($id)
	{
		if (!isset(self::$defaultArticles[$id])) {
			return;
		}

		$template = new Template(self::$defaultArticles[$id], false, null);
		$template->render([
			'zwe_use_addresses' => SettingsRegistry::getInstance()->get('zwe_use_addresses')->getSystemValue(),
			'zwe_use_birthyear' => SettingsRegistry::getInstance()->get('zwe_use_birthyear')->getSystemValue(),
			'zwe_datenschutz_kontakt' => trim(SettingsRegistry::getInstance()->get('zwe_datenschutz_kontakt')->getSystemValue()),
		]);

		$article = new \EmailText();
		$article->fromArray(['text_html' => $template->getContentHtml()]);
		return $article;
	}

	/**
	 * @param $category
	 * @param $id
	 * @return \EmailText|null
	 */
	public function findByCategoryAndId($category, $id)
	{
		if ($category == null) {
			throw new \InvalidArgumentException("Artikel-Kategorie erforderlich!");
		}
		if ($id == null) {
			throw new \InvalidArgumentException("Artikel-ID erforderlich!");
		}

		$descriptor = self::$descriptors[$id];
		if ($descriptor['category'] != $category) {
			return null;
		}

		return $this->find($id);
	}

	/**
	 * @return array
	 */
	public function getDescriptors()
	{
		return self::$descriptors;
	}

	/**
	 * @param $id
	 * @return mixed
	 */
	public function getDescriptor($id)
	{
		if (isset(self::$descriptors[$id])) {
			return self::$descriptors[$id];
		}

		throw new \InvalidArgumentException("Artikel-Deskriptor {$id} existiert nicht!");
	}

	/**
	 * @return array
	 */
	public function getExistingArticleIds()
	{
		$query = \Doctrine_Query::create()
			->select('e.id')
			->from('EmailText e');
		$query->distinct(true);
		$query->setHydrationMode(\Doctrine::HYDRATE_SINGLE_SCALAR);
		$result = $query->execute();
		if (!is_array($result)) {
			return array($result);
		} else {
			return $result;
		}
	}
}
