<?php
namespace ZWE\Images;

use ZWE\Container\ZWEApplicationContainer;
use ZWE\FileStorage\File;

class ImageResourceService
{

	/**
	 * @var ZWEApplicationContainer
	 */
	private $app;

	/**
	 * @var array
	 */
	private $descriptors;

	public function __construct(ZWEApplicationContainer $app)
	{
		$this->app = $app;
		$this->descriptors = array(
			'pdf_header' => array(
				'title' => 'Kopfgrafik PDF',
				'description' => 'Kopfzeilen-Grafik für PDF-Dateien (min. Seitenverhältnis 4:1, Typ png)',
				'mime' => 'image/png',
				'file' => 'pdf_header.png',
				'minratio' => 4.0,
			),
			'pdf_footer' => array(
				'title' => 'Fußgrafik PDF',
				'description' => 'Fußzeilen-Grafik für PDF-Dateien (min. Seitenverhältnis 7:1, Typ png)',
				'mime' => 'image/png',
				'file' => 'pdf_footer.png',
				'minratio' => 7.0,
			),
		);
	}

	public function getDescriptors()
	{
		return $this->descriptors;
	}

	public function find($relativePath)
	{
		try {
			$file = $this->app->getResourceStorage()->retrieve($relativePath);
			$descriptor = $this->findDescriptor($file->getName());
			if ($descriptor != null) {
				return new ImageImpl($file->getAbsolutePath(), $descriptor);
			}
			return null;
		} catch (\Exception $ex) {
			return null;
		}
	}

	/**
	 * @return Image[]
	 */
	public function listImages()
	{
		$images = array();
		$files = $this->app->getResourceStorage()->listFiles();
		foreach ($this->descriptors as $descriptor) {
			$found = false;
			foreach ($files as $file) {
				if ($file->getName() == $descriptor['file']) {
					$images[] = new ImageImpl($file->getAbsolutePath(), $descriptor);
					$found = true;
				}
			}
			if (!$found) {
				$file = $this->app->getResourceStorage()->create($descriptor['file']);
				$images[] = new ImageImpl($file->getAbsolutePath(), $descriptor);
			}
		}
		return $images;
	}

	public function delete($name)
	{
		if ($name == null) {
			throw new \InvalidArgumentException("Name erforderlich!");
		}
		$file = $this->find($name);
		if ($file != null) {
			$file->delete();
		}
	}

	public function upload(File $file, $name = null)
	{
		$descriptor = $this->findDescriptor($name != null ? $name : $file->getName());
		if ($descriptor == null) {
			return array('Kein Deskriptor für Bild-Resource gefunden.');
		}
		$valdidator = new ImageValidator($file, $descriptor);
		if ($valdidator->isValid()) {
			$target = $this->app->getResourceStorage()->create(strtolower($name != null ? $name : $file->getName()), true);
			rename($file->getAbsolutePath(), $target->getAbsolutePath());
			return array();
		} else {
			$file->delete();
			return $valdidator->getErrors();
		}
	}

	private function findDescriptor($name)
	{
		foreach ($this->descriptors as $key => $descriptor) {
			if ($descriptor['file'] == $name) {
				return $descriptor;
			}
		}
		return null;
	}
}
