<?php
namespace ZWE\FileStorage;

use ZWE\Container\ZWEConfigurationSource;

class FileStorageImpl implements FileStorage
{

	/**
	 * @var ZWEConfigurationSource
	 */
	private $configSource;

	private $realPath;

	public function __construct(ZWEConfigurationSource $configSource, $workspacePath)
	{
		if ($workspacePath == null) {
			throw new \InvalidArgumentException("Workspace-Pfad darf nicht NULL sein!");
		}
		$this->configSource = $configSource;

		if (!is_dir($this->configSource["zwe.workspace"])) {
			throw new FileNotFoundException($this->configSource["zwe.workspace"]);
		}
		// @codeCoverageIgnoreStart
		if (!@is_readable($this->configSource["zwe.workspace"])) {
			throw new FileNotReadableException($this->configSource["zwe.workspace"]);
		}
		// @codeCoverageIgnoreEnd

		$path = $this->configSource["zwe.workspace"] . DIRECTORY_SEPARATOR . $workspacePath;
		if (!is_dir($path)) {
			@mkdir($path, 0777, true);
			if (!is_dir($path)) {
				throw new \RuntimeException("Pfad {$path} kann nicht erstellt werden!");
			}
		}
		$this->realPath = realpath($path);
	}

	/**
	 * @param $relativePath
	 * @param $overwrite
	 * @return File
	 */
	public function create($relativePath, $overwrite = false)
	{
		$absolutePath = $this->getAbsolutePath($relativePath);
		if (!$overwrite) {
			@clearstatcache(true, $absolutePath);
			if (@is_file($absolutePath)) {
				throw new FileAlreadyExistsException($absolutePath);
			}
		}
		return new FileImpl($absolutePath);
	}

	/**
	 * @param $relativePath
	 * @return bool
	 */
	public function exists($relativePath)
	{
		return @is_file($this->getAbsolutePath($relativePath));
	}

	/**
	 * @return File[]
	 */
	public function listFiles()
	{
		$files = array();
		foreach (new \DirectoryIterator($this->realPath) as $fileInfo) {
			if ($fileInfo->isDot()) {
				continue;
			} elseif ($fileInfo->isDir()) {
				continue;
			} elseif ($fileInfo->isFile()) {
				$files[] = new FileImpl($fileInfo->getRealPath());
			}
		}
		return $files;
	}

	/**
	 * @return File
	 */
	public function retrieve($relativePath)
	{
		$absolutePath = $this->getAbsolutePath($relativePath);
		@clearstatcache(true, $absolutePath);
		if (!@is_file($absolutePath)) {
			throw new FileNotFoundException($absolutePath);
		}
		// @codeCoverageIgnoreStart
		if (!@is_readable($absolutePath)) {
			throw new FileNotReadableException($absolutePath);
		}
		// @codeCoverageIgnoreEnd
		return new FileImpl($absolutePath);
	}

	private function getAbsolutePath($relativePath)
	{
		return $this->realPath . DIRECTORY_SEPARATOR . $relativePath;
	}

}
