<?php
namespace ZWE\FileStorage;

use Carbon\Carbon;

class FileImpl implements File
{

	/**
	 * @var string
	 */
	private $absolutePath;

	/**
	 * @param bool
	 */
	private $metaDataRead = false;

	/**
	 * @var int
	 */
	private $size;

	/**
	 * @var \DateTime
	 */
	private $modified;

	/**
	 * @var string
	 */
	private $mimeType;

	public function __construct($absolutePath)
	{
		$this->absolutePath = $absolutePath;
	}

	public function delete()
	{
		@unlink($this->getAbsolutePath());
	}

	public function getAbsolutePath()
	{
		return $this->absolutePath;
	}

	public function getMimeType()
	{
		$this->readMetaData();
		return $this->mimeType;
	}

	public function getName()
	{
		return basename($this->getAbsolutePath());
	}

	public function getSize()
	{
		$this->readMetaData();
		return $this->size;
	}

	public function getModified()
	{
		$this->readMetaData();
		return $this->modified;
	}

	public function exists()
	{
		@clearstatcache(true, $this->getAbsolutePath());
		return @file_exists($this->getAbsolutePath());
	}

	/**
	 * @codeCoverageIgnore
	 */
	public function send()
	{
		if ($fp = $this->open()) {
			$this->sendViewHeaders();
			header('Content-disposition: attachment; filename="' . $this->getName() . '"', true);
			@fpassthru($fp);
			$this->close($fp);
		} else {
			$this->error404();
		}
	}

	/**
	 * @codeCoverageIgnore
	 */
	public function view()
	{
		if ($fp = $this->open()) {
			$this->sendViewHeaders();
			header('Content-disposition: inline; filename="' . $this->getName() . '"', true);
			@fpassthru($fp);
			$this->close($fp);
		} else {
			$this->error404();
		}
	}

	/**
	 * @return string
	 */
	public function read()
	{
		return @file_get_contents($this->absolutePath);
	}

	/**
	 * @param $contents
	 * @return void
	 */
	public function write($contents)
	{
		@file_put_contents($this->absolutePath, $contents);
	}

	/**
	 * @codeCoverageIgnore
	 */
	protected function sendViewHeaders()
	{
		$this->readMetaData();
		header('Content-length: ' . $this->size, true);
		header('Content-type: ' . $this->mimeType, true);
		header('Cache-control: private, max-age: 0, must-revalidate', true);
	}

	/**
	 * @codeCoverageIgnore
	 */
	private function error404()
	{
		header("HTTP/1.0 404 Not Found", true);
	}

	private function open()
	{
		if ($this->exists()) {
			return @fopen($this->getAbsolutePath(), 'r');
		} else {
			return null;
		}
	}

	private function close($fp)
	{
		if ($fp) {
			@fclose($fp);
		}
	}

	private function readMetaData()
	{
		if ($this->exists()) {
			if (!$this->metaDataRead) {
				@clearstatcache(true, $this->getAbsolutePath());
				$this->size = filesize($this->getAbsolutePath());
				$this->mimeType = $this->readMimeType();
				$this->modified = Carbon::createFromTimestamp(filemtime($this->getAbsolutePath()));
				$this->metaDataRead = true;
			}
		} else {
			$this->size = 0;
			$this->mimeType = null;
		}
	}

	private function readMimeType()
	{
		$pi = pathinfo($this->getAbsolutePath());
		switch (strtolower($pi['extension'])) {
			case 'doc':
				return 'application/msword';
			case 'docx':
				return 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
			case 'jpg':
			case 'jpeg':
				return 'image/jpeg ';
			case 'pdf';
				return 'application/pdf';
			case 'png':
				return 'image/png';
			case 'xls':
				return 'application/vnd.ms-excel';
			case 'xlsx':
				return 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
			default:
				return 'application/octet-stream';
		}
	}
}
