<?php
namespace ZWE\Export\Freigaben;

use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Conditional;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Style\Style;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use ZWE\Export\AbstractExcelExport;
use function ZWE\count_safe;

class ExportPlanungExcelImpl extends AbstractExcelExport implements ExportPlanungExcel
{

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Color
	 */
	private $colorGreen;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Color
	 */
	private $colorYellow;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Color
	 */
	private $colorDarkRed;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Color
	 */
	private $colorWhite;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Color
	 */
	private $colorDarkGreen;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Color
	 */
	private $colorDarkOrange;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleFreigabeLizenzPasst;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleFreigabeLizenzPasstNicht;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleTurnierHeaderDatum;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleTurnierHeader;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleTurnierHeaderBold;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleWrHeaderLeft;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleWrHeaderCenter;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleWrHeaderCenterBorderRight;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleCenter;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleJudgeCount;

	/**
	 * @var \PhpOffice\PhpSpreadsheet\Style\Style
	 */
	private $styleWrCenterBorderRight;

	private $start = 0;

	public function createDocument(\DateTime $from, \DateTime $to, $eventMandanten = null, $wrMandanten = null, $ltvs = null)
	{
		$start = time();

		$sheet = new Worksheet($this->excel, 'Planung');
		$this->excel->addSheet($sheet);

		// colors
		$this->colorGreen = new Color(Color::COLOR_GREEN);
		$this->colorYellow = new Color(Color::COLOR_YELLOW);
		$this->colorWhite = new Color(Color::COLOR_WHITE);
		$this->colorDarkGreen = new Color(Color::COLOR_DARKGREEN);
		$this->colorDarkRed = new Color(Color::COLOR_DARKRED);
		$this->colorDarkOrange = new Color();
		$this->colorDarkOrange->setARGB('FF800000');

		// shared styles

		$this->styleFreigabeLizenzPasst = new Style();
		$this->styleFreigabeLizenzPasst->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER
			),
			'fill' => array(
				'fillType' => Fill::FILL_SOLID,
				'color' => array('argb' => Color::COLOR_GREEN)
			)
		));

		$this->styleFreigabeLizenzPasstNicht = new Style();
		$this->styleFreigabeLizenzPasstNicht->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER
			),
			'fill' => array(
				'fillType' => Fill::FILL_SOLID,
				'color' => array('argb' => Color::COLOR_YELLOW)
			)
		));

		$this->styleTurnierHeader = new Style();
		$this->styleTurnierHeader->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER,
				'vertical' => Alignment::VERTICAL_TOP
			)
		));

		$this->styleTurnierHeaderBold = new Style();
		$this->styleTurnierHeaderBold->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER,
				'vertical' => Alignment::VERTICAL_TOP
			),
			'font' => array(
				'bold' => true
			)
		));

		$this->styleTurnierHeaderDatum = new Style();
		$this->styleTurnierHeaderDatum->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER,
				'vertical' => Alignment::VERTICAL_TOP
			),
			'font' => array(
				'bold' => true
			),
			'numberFormat' => array(
				'formatCode' => NumberFormat::FORMAT_DATE_DDMMYYYY
			)
		));

		$this->styleWrHeaderLeft = new Style();
		$this->styleWrHeaderLeft->applyFromArray(array(
			'borders' => array(
				'bottom' => array(
					'borderStyle' => Border::BORDER_MEDIUM,
					'color' => array(
						'rgb' => '000000'
					)
				)
			),
			'font' => array(
				'bold' => true
			)
		));

		$this->styleWrHeaderCenter = new Style();
		$this->styleWrHeaderCenter->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER
			),
			'borders' => array(
				'bottom' => array(
					'borderStyle' => Border::BORDER_MEDIUM,
					'color' => array(
						'rgb' => '000000'
					)
				)
			),
			'font' => array(
				'bold' => true
			)
		));

		$this->styleCenter = new Style();
		$this->styleCenter->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER
			)
		));

		$this->styleJudgeCount = new Style();
		$this->styleJudgeCount->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER
			),
			'borders' => array(
				'bottom' => array(
					'borderStyle' => Border::BORDER_MEDIUM,
					'color' => array(
						'rgb' => '000000'
					)
				)
			),
		));

		$this->styleWrCenterBorderRight = new Style();
		$this->styleWrCenterBorderRight->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER
			),
			'borders' => array(
				'right' => array(
					'borderStyle' => Border::BORDER_MEDIUM,
					'color' => array(
						'rgb' => '000000'
					)
				)
			),
		));

		$this->styleWrHeaderCenterBorderRight = new Style();
		$this->styleWrHeaderCenterBorderRight->applyFromArray(array(
			'alignment' => array(
				'horizontal' => Alignment::HORIZONTAL_CENTER
			),
			'borders' => array(
				'right' => array(
					'borderStyle' => Border::BORDER_MEDIUM,
					'color' => array(
						'rgb' => '000000'
					)
				),
				'bottom' => array(
					'borderStyle' => Border::BORDER_MEDIUM,
					'color' => array(
						'rgb' => '000000'
					)
				)
			),
			'font' => array(
				'bold' => true
			)
		));

		// create document
		$additionalRows = 0;

		// events
		$events = $this->container->getEventRepo()->findBetween($from, $to, true, true, $eventMandanten);
		$teamCount = 0;
		$col = 8;
		$row = 1;
		if ($events->count() > 0) {
			foreach ($events as $event) {
				foreach ($event->WRTeam as $team) {
					$rows = $this->addTurnierHeader($sheet, $col, $row, $team);
					$col++;
					if ($rows > $additionalRows) {
						$additionalRows = $rows;
					}
					$teamCount++;
				}
			}
		} else {
			$this->finish($sheet, true);
			return;
		}

		// border header
		$col = 7;
		$row = 1;
		$sheet->duplicateStyle($this->styleWrCenterBorderRight, $this->getRangeAddress($col, $row, $col, $row + 4 + $additionalRows));

		// extract wr data
		$wrs = array();
		$daten = array();

		$freigaben = $this->container->getAvailabilityRepo()->getAvailabilitiesBetween($from, $to, $wrMandanten, $ltvs);
		if ($freigaben->count() > 0) {
			foreach ($freigaben as $freigabe) {
				$daten[$freigabe->User->id][$freigabe->datum] = true;
				if (!isset($wrs[$freigabe->User->id])) {
					$wrs[$freigabe->User->id] = $freigabe->User;
				}
			}
		}
		else {
			$this->finish($sheet, true);
			return;
		}

		uasort($wrs, function (\User $a, \User $b) {
			if ($a->Club->id == $b->Club->id) {
				return strcmp(sprintf('%1$s %2$s', $a->surname, $a->name), sprintf('%1$s %2$s', $b->surname, $b->name));
			} else {
				return strcmp($a->Club->name, $b->Club->name);
			}
		});

		// wr header
		$col = 1;
		$row = 5 + $additionalRows;

		$sheet->setCellValueByColumnAndRow($col, $row, 'Name');
		$sheet->getColumnDimensionByColumn($col)->setAutoSize(true);
		$sheet->duplicateStyle($this->styleWrHeaderLeft, $this->getCellAddress($col, $row));

		$sheet->setCellValueByColumnAndRow($col + 1, $row, 'St');
		$sheet->getColumnDimensionByColumn($col + 1)->setAutoSize(true);
		$sheet->setCellValueByColumnAndRow($col + 2, $row, 'Lat');
		$sheet->getColumnDimensionByColumn($col + 2)->setAutoSize(true);
		$sheet->duplicateStyle($this->styleWrHeaderCenter, $this->getRangeAddress($col + 1, $row, $col + 2, $row));

		$sheet->setCellValueByColumnAndRow($col + 3, $row, 'Verein');
		$sheet->getColumnDimensionByColumn($col + 3)->setAutoSize(true);
		$sheet->duplicateStyle($this->styleWrHeaderLeft, $this->getCellAddress($col + 3, $row));

		$sheet->setCellValueByColumnAndRow($col + 4, $row, 'E');
		$sheet->getColumnDimensionByColumn($col + 4)->setWidth(3);
		$sheet->setCellValueByColumnAndRow($col + 5, $row, 'F');
		$sheet->getColumnDimensionByColumn($col + 5)->setWidth(3);
		$sheet->duplicateStyle($this->styleWrHeaderCenter, $this->getRangeAddress($col + 4, $row, $col + 5, $row));

		$sheet->setCellValueByColumnAndRow($col + 6, $row, 'P');
		$sheet->getColumnDimensionByColumn($col + 6)->setWidth(3);
		$sheet->duplicateStyle($this->styleWrHeaderCenterBorderRight, $this->getCellAddress($col + 6, $row));

		// wr
		$row = 6 + $additionalRows;
		foreach ($wrs as $wr) {
			$sheet->setCellValueByColumnAndRow($col, $row, sprintf('%1$s, %2$s', $wr->surname, $wr->name));

			$sheet->setCellValueByColumnAndRow($col + 1, $row, $wr->LicenseStEffective->shortname);
			$sheet->setCellValueByColumnAndRow($col + 2, $row, $wr->LicenseLatEffective->shortname);
			$sheet->duplicateStyle($this->styleCenter, $this->getRangeAddress($col + 1, $row, $col + 2, $row));

			if ($wr->Club->LTV->esv_key) {
				$sheet->setCellValueByColumnAndRow($col + 3, $row, $wr->Club->name . ' (' . $wr->Club->LTV->esv_key . ')');
			} else {
				$sheet->setCellValueByColumnAndRow($col + 3, $row, $wr->Club->name . ' (' . $wr->Club->Country->alpha3 . ')');
			}

			$sheet->setCellValueByColumnAndRow($col + 4, $row, $wr->getStatistics()->confirmed);
			if (isset($daten[$wr->id])) {
				$sheet->setCellValueByColumnAndRow($col + 5, $row, count_safe($daten[$wr->id]));
			} else {
				$sheet->setCellValueByColumnAndRow($col + 5, $row, 0);
			}
			$sheet->duplicateStyle($this->styleCenter, $this->getRangeAddress($col + 4, $row, $col + 5, $row));

			$sheet->setCellValueByColumnAndRow($col + 6, $row, '=COUNTA(' . $this->getRangeAddress($col + 7, $row, $col + 7 + $teamCount - 1, $row) . ')');
			$sheet->duplicateStyle($this->styleWrCenterBorderRight, $this->getCellAddress($col + 6, $row));

			$row++;
		}

		// matrix
		$col = 8;
		$row = 6 + $additionalRows;
		// freeze the pane
		$sheet->freezePane($this->getCellAddress($col, $row));
		foreach ($wrs as $wr) {
			$currentCol = $col;
			foreach ($events as $event) {
				foreach ($event->WRTeam as $team) {
					$cellAddress = $this->getCellAddress($currentCol, $row);
					if (isset($daten[$wr->id][$event->datum])) {
						if ($wr->isLicenseSufficient($team)) {
							$sheet->duplicateStyle($this->styleFreigabeLizenzPasst, $cellAddress);
						} else {
							$sheet->duplicateStyle($this->styleFreigabeLizenzPasstNicht, $cellAddress);
						}
					}
					$currentCol++;
				}
			}
			$row++;
		}


		// einsatzanzahl
		$row = 5 + $additionalRows;
		$col = 8;
		$wrcount = count_safe($wrs);
		foreach ($events as $event) {
			foreach ($event->WRTeam as $team) {
				$cellAddress = $this->getCellAddress($col, $row);
				$sheet->setCellValueByColumnAndRow($col, $row, '=COUNTA(' . $this->getRangeAddress($col, $row + 1, $col, $row + $wrcount) . ')');
				$sheet->setConditionalStyles($cellAddress, $this->createConditionalsForJudgecount($team->judgecount));
				$sheet->duplicateStyle($this->styleJudgeCount, $cellAddress);
				$col++;
			}
		}

		// debug
		//$sheet->setCellValueByColumnAndRow(0, 1, (float)memory_get_usage(true) / 1024 / 1024 . " MB");
		//$sheet->setCellValueByColumnAndRow(0, 2, (float)memory_get_peak_usage(false) / 1024 / 1024 . " MB");
		//$end = microtime(true);
		//$sheet->setCellValueByColumnAndRow(0, 3, $end - $start . " sec");

		$this->finish($sheet);
	}

	/**
	 * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
	 * @param $col
	 * @param $firstRow
	 * @param \WRTeam $team
	 * @return int
	 */
	public function addTurnierHeader(Worksheet $sheet, $col, $firstRow, \WRTeam $team)
	{
		$sheet->setCellValueByColumnAndRow($col, $firstRow, Date::PHPToExcel($team->Event->getDateTimeObject('datum')));
		$sheet->duplicateStyle($this->styleTurnierHeaderDatum, $this->getRangeAddress($col, $firstRow, $col, $firstRow));
		$sheet->setCellValueByColumnAndRow($col, $firstRow + 1, $this->abbreviate($team->Event->Location->name, 20));
		$sheet->setCellValueByColumnAndRow($col, $firstRow + 2, $this->abbreviate($team->Event->Club->name, 20));
		$sheet->setCellValueByColumnAndRow($col, $firstRow + 3, $team->name . " (" . $team->judgecount . ")");
		$sheet->getColumnDimensionByColumn($col)->setWidth(20);
		$sheet->duplicateStyle($this->styleTurnierHeaderBold, $this->getRangeAddress($col, $firstRow + 1, $col, $firstRow + 3));

		$additionalRows = 0;
		foreach ($team->Competitions as $competition) {
			$sheet->setCellValueByColumnAndRow($col, $firstRow + 4 + $additionalRows, sprintf('%1$s %2$s %3$s', $competition->Startgroup->name, $competition->Startclass->name, $competition->Dancetype->name));
			$additionalRows++;
		}
		$sheet->duplicateStyle($this->styleTurnierHeader, $this->getRangeAddress($col, $firstRow + 4, $col, $firstRow + 4 + $additionalRows));

		return $additionalRows;
	}

	private function getCellAddress($col, $row)
	{
		return Coordinate::stringFromColumnIndex($col) . $row;
	}

	private function getRangeAddress($col1, $row1, $col2, $row2)
	{
		return $this->getCellAddress($col1, $row1) . ':' . $this->getCellAddress($col2, $row2);
	}

	private function createConditionalStyleForCellIsLessThan($value)
	{
		$cond = new Conditional();
		$cond->setConditionType(Conditional::CONDITION_CELLIS);
		$cond->setOperatorType(Conditional::OPERATOR_LESSTHAN);
		$cond->addCondition($value);
		$cond->getStyle()->getFont()->setColor($this->colorWhite);
		$cond->getStyle()->getFont()->setBold(true);
		$cond->getStyle()->getFill()->setFillType(Fill::FILL_SOLID);
		$cond->getStyle()->getFill()->setStartColor($this->colorDarkOrange);
		$cond->getStyle()->getFill()->setEndColor($this->colorDarkOrange);
		return $cond;
	}

	private function createConditionalStyleForCellIsEqualTo($value)
	{
		$cond = new Conditional();
		$cond->setConditionType(Conditional::CONDITION_CELLIS);
		$cond->setOperatorType(Conditional::OPERATOR_EQUAL);
		$cond->addCondition($value);
		$cond->getStyle()->getFont()->setColor($this->colorWhite);
		$cond->getStyle()->getFont()->setBold(true);
		$cond->getStyle()->getFill()->setFillType(Fill::FILL_SOLID);
		$cond->getStyle()->getFill()->setStartColor($this->colorDarkGreen);
		$cond->getStyle()->getFill()->setEndColor($this->colorDarkGreen);
		return $cond;
	}

	private function createConditionalStyleForCellIsGreaterThan($value)
	{
		$cond = new Conditional();
		$cond->setConditionType(Conditional::CONDITION_CELLIS);
		$cond->setOperatorType(Conditional::OPERATOR_GREATERTHAN);
		$cond->addCondition($value);
		$cond->getStyle()->getFont()->setColor($this->colorWhite);
		$cond->getStyle()->getFont()->setBold(true);
		$cond->getStyle()->getFill()->setFillType(Fill::FILL_SOLID);
		$cond->getStyle()->getFill()->setStartColor($this->colorDarkRed);
		$cond->getStyle()->getFill()->setEndColor($this->colorDarkRed);
		return $cond;
	}

	private function createConditionalsForJudgecount($judgecount)
	{
		return array($this->createConditionalStyleForCellIsLessThan($judgecount),
			$this->createConditionalStyleForCellIsEqualTo($judgecount),
			$this->createConditionalStyleForCellIsGreaterThan($judgecount));
	}

	private function abbreviate($string, $length)
	{
		if ($string != null) {
			if (strlen($string) <= $length) {
				return $string;
			} else {
				return substr($string, 0, $length - 1) . '…';
			}
		} else {
			return null;
		}
	}

	public function send($format = self::FORMAT_XLSX, $baseFilename = null)
	{
		parent::send($format, $baseFilename);
	}

	private function finish($sheet, $empty = false)
	{
		if($empty) {
			$this->excel->removeSheetByIndex(0);
			$sheet = new Worksheet($this->excel, 'Planung');
			$this->excel->addSheet($sheet);
			$sheet->setCellValue('A1', 'keine Daten');
		}
		$this->excel->setActiveSheetIndex(0);
		$this->created = true;
	}
}
