<?php

namespace ZWE\ESV;

use Carbon\Carbon;
use JMS\Serializer\SerializerInterface;
use Tanzsport\ESV\API\Client;
use Tanzsport\ESV\API\Model\Club;
use Tanzsport\ESV\API\Model\Funktionaer\Funktionaer;
use ZWE\ESV\ExportFormat\Model\ZEFWertungsrichterModel;
use ZWE\FileStorage\FileStorage;
use ZWE\Repository\ClubRepository;
use ZWE\Repository\CountryRepository;
use ZWE\Repository\LicenseRepository;
use ZWE\Repository\LTVRepository;
use ZWE\Repository\UserRepository;
use function ZWE\count_safe;

class ESVService
{

	private static $FUNKTIONAERE_CACHE = 'funktionaere.ser';

	/**
	 * @var Client
	 */
	private $client;

	/**
	 * @var FileStorage
	 */
	private $storage;

	/**
	 * @var DTVNummerGenerator
	 */
	private $generator;

	/**
	 * @var UserRepository
	 */
	private $userRepository;

	/**
	 * @var ClubRepository
	 */
	private $clubRepository;

	/**
	 * @var LTVRepository
	 */
	private $ltvRepository;

	/**
	 * @var LicenseRepository
	 */
	private $licenseRepsitory;

	/**
	 * @var CountryRepository
	 */
	private $countryRepository;

	/**
	 * @var SerializerInterface
	 */
	private $serializer;

	/**
	 * @var array
	 */
	private $cache = [];

	public function __construct(Client $client, FileStorage $storage, DTVNummerGenerator $generator,
		UserRepository $userRepository, ClubRepository $clubRepository, LTVRepository $ltvRepository,
		LicenseRepository $licenseRepository, CountryRepository $countryRepository, SerializerInterface $serializer)
	{
		$this->client = $client;
		$this->storage = $storage;
		$this->generator = $generator;
		$this->userRepository = $userRepository;
		$this->clubRepository = $clubRepository;
		$this->ltvRepository = $ltvRepository;
		$this->licenseRepsitory = $licenseRepository;
		$this->countryRepository = $countryRepository;
		$this->serializer = $serializer;
	}

	/**
	 * @return ESVReportVerein
	 */
	public function vereinsDatenAbgleichen($normalisieren = false)
	{
		$funktionaere = $this->funktionaereLaden();

		/**
		 * @var Club[] $esvVereine
		 */
		$esvVereine = [];
		foreach ($funktionaere as $funktionaer) {
			if (!isset($esvVereine[$funktionaer->club->id])) {
				$esvVereine[$funktionaer->club->id] = $funktionaer->club;
			}
		}

		/**
		 * @var $zweVereine \Club[]
		 */
		$zweVereine = [];
		foreach ($this->clubRepository->findAll() as $zweVerein) {
			/**
			 * @var $zweVerein \Club
			 */
			if (!in_array($zweVerein->LTV->id, ['EXT', '_none_'])) {
				$zweVereine[] = $zweVerein;
			}
		}

		/**
		 * @var $reports ESVReportVerein
		 */
		$reports = [];
		foreach ($zweVereine as $zweVerein) {
			/**
			 * @var $zweVerein \Club
			 */
			if (isset($esvVereine[$zweVerein->vnum_dtv])) {
				$report = new ESVReportVerein($zweVerein, $esvVereine[$zweVerein->vnum_dtv], $normalisieren);
				if ($report->isAusgeben()) {
					$reports[] = $report;
				}
			}
		}

		return $reports;
	}

	/**
	 * @param ESVAbweichung[] $korrekturen
	 */
	public function vereinsDatenKorrigieren(array $korrekturen)
	{
		foreach ($korrekturen as $id => $abweichungen) {
			$zweVerein = $this->clubRepository->find($id);
			$changed = false;
			/**
			 * @var $abweichung ESVAbweichung
			 * @var $zweVerein \Club
			 */
			if ($zweVerein) {
				foreach ($abweichungen as $abweichung) {
					switch ($abweichung->getFeld()) {
						case 'verein_name':
						case 'verein_nname':
							$zweVerein->name = $abweichung->getNeuerWert();
							$changed = true;
							break;
						case 'verein_ltv':
							$ltv = $this->ltvRepository->findByEsvKey($abweichung->getNeuerWert());
							if (!$ltv) {
								throw new \RuntimeException("LTV mit ESV-Schlüssel {$abweichung->getNeuerWert()} nicht gefunden!");
							}
							$zweVerein->LTV = $ltv;
							$changed = true;
							break;
						default:
							throw new \RuntimeException("Feld {$abweichung->getFeld()} wird nicht unterstützt!");
					}
				}
				if ($changed) {
					$zweVerein->save();
				}
			}
		}
	}

	public function lizenzDatenAbgleich()
	{
		$abweichungen = [];

		$index = [];
		$funktionaere = $this->funktionaereLaden();
		foreach ($funktionaere as $funktionaer) {
			if (!isset($index[$funktionaer->id])) {
				$index[$funktionaer->id] = $funktionaer;
			}
		}

		$zeitstempel = new \DateTime();

		/**
		 * @var $user \User
		 */
		foreach ($this->userRepository->findAll() as $user) {
			if (!$user->isWR() || $user->Club->LTV->id == 'EXT') {
				continue;
			}

			if ($user->isActive() || $user->isUnlicensed()) {
				if (isset($index[$user->dtvnr])) {
					$user->setEsvDaten($index[$user->dtvnr], $zeitstempel);
					$user->save();
					$abweichung = new ESVReportLizenztraeger($user, true);
					if ($abweichung->isAusgeben()) {
						$abweichungen[] = $abweichung;
					}
				} else {
					if ($user->esv_check) {
						$lastCheck = $user->getDateTimeObject('esv_check');
						$user->last_license_year = intval($lastCheck->format('Y'));
						$user->save();
					}
					$abweichung = new ESVReportLizenztraeger($user, false);
					if ($abweichung->isAusgeben()) {
						$abweichungen[] = $abweichung;
					}
				}
			}
		}

		return $abweichungen;
	}

	/**
	 * @param ESVAbweichung[] $korrekturen
	 */
	public function lizenzDatenKorrigieren(array $korrekturen, $normalisieren = false)
	{
		$lizenzen = [];
		foreach ($this->licenseRepsitory->getAllLicenseObjects(true, null, [\License::TYPE_EINZEL]) as $zweLizenz) {
			$lizenzen[$zweLizenz->shortname] = $zweLizenz;
		}

		foreach ($korrekturen as $id => $abweichungen) {
			$zweWr = $this->userRepository->find($id);
			$changed = false;
			$licenseChanged = false;
			/**
			 * @var $abweichung ESVAbweichung
			 * @var $zweWr \User
			 */
			if ($zweWr) {
				foreach ($abweichungen as $abweichung) {
					switch ($abweichung->getFeld()) {
						case 'person_vorname':
							$zweWr->name = $abweichung->getNeuerWert();
							$changed = true;
							break;
						case 'person_nachname':
							$zweWr->surname = $abweichung->getNeuerWert();
							$changed = true;
							break;
						case 'person_titel':
							$zweWr->title = $abweichung->getNeuerWert();
							$changed = true;
							break;
						case 'person_vnum':
							$zweClub = $this->clubRepository->findByDTVVnum($abweichung->getNeuerWert());
							if ($zweClub == null) {
								$zweClub = $this->fromESVVerein($zweWr->getEsvDaten()->club, $normalisieren);
							}
							$zweWr->Club = $zweClub;
							$changed = true;
							break;
						case 'person_status':
							$zweWr->active = $abweichung->getNeuerWert();
							$changed = true;
							break;
						case 'person_lstd':
							if (isset($lizenzen[$abweichung->getNeuerWert()])) {
								$zweWr->LicenseSt = $lizenzen[$abweichung->getNeuerWert()];
								$changed = true;
								$licenseChanged = true;
							}
							break;
						case 'person_llat':
							if (isset($lizenzen[$abweichung->getNeuerWert()])) {
								$zweWr->LicenseLat = $lizenzen[$abweichung->getNeuerWert()];
								$changed = true;
								$licenseChanged = true;
							}
							break;
						case 'person_leinschraenkung':
							if (isset($lizenzen[$abweichung->getNeuerWert()])) {
								$zweWr->LicenseEffective = $lizenzen[$abweichung->getNeuerWert()];
								$changed = true;
								$licenseChanged = true;
							}
							break;
						default:
							throw new \RuntimeException("Feld {$abweichung->getFeld()} wird nicht unterstützt!");
					}
				}
				if ($changed) {
					if ($licenseChanged) {
						$zweWr->updateEffectiveLicenses();
					}
					$zweWr->save();
				}
			}
		}
	}

	/**
	 * @param Club $esvVerein
	 * @return \Club
	 */
	public function fromESVVerein(Club $esvVerein, $normalisieren = false)
	{
		$zweVerein = new \Club;
		$zweVerein->vnum_dtv = $esvVerein->id;
		if ($normalisieren) {
			$zweVerein->name = $this->vereinsNameNormalisieren($esvVerein->name);
		} else {
			$zweVerein->name = $esvVerein->name;
		}
		$zweVerein->ausrichter = 0;
		$zweVerein->has_wr = 1;
		$ltv = $this->ltvRepository->loadByEsvKey($esvVerein->ltv->name);
		$zweVerein->LTV = $ltv;
		$zweVerein->Owner = $ltv->DefaultMandator;
		$zweVerein->Country = $this->countryRepository->findByISOAlpha3('DEU');
		$zweVerein->save();
		return $zweVerein;
	}

	private function vereinsNameNormalisieren($input)
	{
		return trim(preg_replace('/e\.\s?v\.$|e\.\s?v\.,? |ev$/i', '', $input));
	}

	public function dtvNummerAbgleichen()
	{
		$report = new AbgleichDTVNummerReport();

		$index = [];
		$funktionaere = $this->funktionaereLaden();
		foreach ($funktionaere as $funktionaer) {
			if ($funktionaer->lizenzNr > 0) {
				if (!isset($index[$funktionaer->lizenzNr])) {
					$index[$funktionaer->lizenzNr] = $funktionaer;
				} else {
					$report->doppelteLizenznummern[] = $funktionaer;
				}
			}
		}

		/**
		 * @var $user \User
		 */
		foreach ($this->userRepository->findAll() as $user) {
			$report->userCount++;
			if ($user->isWR()) {
				$report->wrCount++;
			} else {
				$report->systemCount++;
			}

			if ($user->dtvnr) {
				$report->alreadySetCount++;
				continue;
			} else {
				if ($user->isWR()) {
					if (!$user->isActive() && !$user->isUnlicensed()) {
						$report->inaktivWrCount++;
					} else {
						if (!$user->Club->LTV->isESV()) {
							$user->dtvnr = $this->generator->naechsteNummerGenerieren('WR', 1001)->__toString();
							$user->save();
							$report->setCountWrExt++;
						} else {
							if (isset($index[$user->liznr])) {
								$user->dtvnr = $index[$user->liznr]->id;
								$report->setCountWrInt++;
								$user->save();
							} else {
								$report->nichtGefunden[] = $user;
							}
						}
					}
				} else {
					if (!$user->isActive()) {
						$report->inaktivSystemCount++;
					} else {
						$user->dtvnr = $this->generator->naechsteNummerGenerieren('ZE', 1001)->__toString();
						$user->save();
						$report->setCountSystem++;
					}
				}
			}
		}

		return $report;
	}

	/**
	 * @return \Tanzsport\ESV\API\Model\Funktionaer\Funktionaer[]
	 */
	private function funktionaereLaden()
	{
		if ($this->storage->exists(self::$FUNKTIONAERE_CACHE)) {
			$cache = $this->storage->retrieve(self::$FUNKTIONAERE_CACHE);
			if ($cache->getModified()->addDay(1)->lt(Carbon::now())) {
				$cache->delete();
			} else {
				return unserialize($this->storage->retrieve(self::$FUNKTIONAERE_CACHE)->read());
			}
		}

		$funktionaere = $this->client->getFunktionaerResource()->findeAlleFunktionaere();
		$cache = $this->storage->create(self::$FUNKTIONAERE_CACHE);
		$cache->write(serialize($funktionaere));

		return $funktionaere;
	}


	/**
	 * @param $id DTV-Nummer
	 * @return null|Funktionaer
	 */
	public function funktionaerFinden($id)
	{
		if (!$id) {
			throw new \InvalidArgumentException('ID erforderlich!');
		}

		if (isset($this->cache[$id])) {
			return $this->cache[$id];
		}

		$funktionaere = $this->funktionaereLaden();
		if ($funktionaere == null || empty($funktionaere)) {
			return null;
		}
		foreach ($funktionaere as $funktionaer) {
			if ($funktionaer->id == $id) {
				$this->cache[$id] = $funktionaer;
				return $funktionaer;
			}
		}

		$funktionaer = $this->client->getFunktionaerResource()->findeFunktionaerNachDtvId($id);
		if ($funktionaer != null) {
			$this->cache[$id] = $funktionaer;
		}
		return $funktionaer;
	}

	/**
	 * @param array $veranstalungsIds
	 * @return ZEFWertungsrichterModel[]
	 */
	public function exportLaden(array $veranstalungsIds)
	{
		$wrs = [];
		if (count_safe($veranstalungsIds) > 0) {
			$query = \Doctrine_Query::create();
			$query->distinct(true);
			$query->select('user.*', 'club.*', 'ltv.*', 'country.*')
				->from('User user')
				->innerJoin('user.Club club')
				->innerJoin('club.LTV ltv')
				->innerJoin('club.Country country')
				->innerJoin('user.Assignments assignment')
				->innerJoin('assignment.WRTeam team')
				->innerJoin('team.Event event')
				->where('assignment.state = ?', \Assignment::STATE_CONFIRMED)
				->andWhereIn('event.id', $veranstalungsIds)
				->orderBy('user.surname ASC, user.name ASC');
			$result = $query->execute();
			foreach ($result as $r) {
				$wrs[] = new ZEFWertungsrichterModel($r, $r->Club);
			}
		}
		return $wrs;
	}

	/**
	 * @param array $veranstaltungsIds
	 * @return string
	 */
	public function exportLadenUndKonvertieren(array $veranstaltungsIds)
	{
		return $this->serializer->serialize($this->exportLaden($veranstaltungsIds), 'json');
	}
}