<?php

namespace ZWE\ESV;

use function ZWE\count_safe;

class ESVReportLizenztraeger
{

	/**
	 * @var \User
	 */
	private $user;

	/**
	 * @var bool
	 */
	private $ausgeben = false;

	/**
	 * @var ESVAbweichung[]
	 */
	private $abweichungen = [];

	public function __construct(\User $user, $esvWasUpdated = true)
	{
		$this->user = $user;

		if ($esvWasUpdated) {
			// nachname
			if ($user->surname != $user->getEsvDaten()->nachname) {
				$this->ausgeben = true;
				$this->abweichungen[] = new ESVAbweichung('person_nachname', $user->surname, $user->getEsvDaten()->nachname, true);
			}

			// vorname
			if ($user->name != $user->getEsvDaten()->vorname) {
				$this->ausgeben = true;
				$this->abweichungen[] = new ESVAbweichung('person_vorname', $user->name, $user->getEsvDaten()->vorname, true);
			}

			// titel
			if ($user->title != $user->getEsvDaten()->titel) {
				$this->ausgeben = true;
				$this->abweichungen[] = new ESVAbweichung('person_titel', $user->title, $user->getEsvDaten()->titel, true);
			}

			// verein
			if ($user->Club->vnum_dtv != $user->getEsvDaten()->club->id) {
				$this->ausgeben = true;
				$this->abweichungen[] = new ESVAbweichung('person_vnum', $user->Club->vnum_dtv, $user->getEsvDaten()->club->id,
					true, $user->Club->name . ' (#' . $user->Club->vnum_dtv . ')',
					$user->getEsvDaten()->club->name . ' (#' . $user->getEsvDaten()->club->id . ')');
			}

			$esvLizenzStandard = ESVLizenzen::konvertiereHoechsteLizenzStd($user->getEsvDaten()->lizenzen);
			$esvLizenzLatein = ESVLizenzen::konvertiereHoechsteLizenzLat($user->getEsvDaten()->lizenzen);

			// status und lizenzen
			if ($esvLizenzStandard == null || $esvLizenzLatein == null) {
				if (!$user->isUnlicensed()) {
					$this->ausgeben = true;
					$this->abweichungen[] = new ESVAbweichung('person_status', $user->active, \User::STATE_UNLICENSED, true, \User::getStatusAsTextStatic($user->active), \User::getStatusAsTextStatic(\User::STATE_UNLICENSED));
				}
			} else {
				if ($user->isUnlicensed()) {
					$this->ausgeben = true;
					$this->abweichungen[] = new ESVAbweichung('person_status', $user->active, \User::STATE_ACTIVE, true, \User::getStatusAsTextStatic($user->active), \User::getStatusAsTextStatic(\User::STATE_ACTIVE));
				}

				$abweichendeLizenzen = [];

				if ($user->LicenseSt->shortname != $esvLizenzStandard) {
					$this->ausgeben = true;
					$this->abweichungen[] = new ESVAbweichung('person_lstd', $user->LicenseSt->shortname, $esvLizenzStandard, true);
					$abweichendeLizenzen[] = 'Std';
				}

				if ($user->LicenseLat->shortname != $esvLizenzLatein) {
					$this->ausgeben = true;
					$this->abweichungen[] = new ESVAbweichung('person_llat', $user->LicenseLat->shortname, $esvLizenzLatein, true);
					$abweichendeLizenzen[] = 'Lat';
				}

				if (count_safe($abweichendeLizenzen) > 0) {
					if ($user->LicenseEffective->shortname != '-') {
						if (count_safe($abweichendeLizenzen) == 1) {
							if (in_array('Std', $abweichendeLizenzen)) {
								$this->abweichungen[] = new ESVAbweichung('person_llat', $user->LicenseLat->shortname, $esvLizenzLatein, false);
							}
							if (in_array('Lat', $abweichendeLizenzen)) {
								$this->abweichungen[] = new ESVAbweichung('person_lstd', $user->LicenseSt->shortname, $esvLizenzStandard, false);
							}
						}
						$this->abweichungen[] = new ESVAbweichung('person_leinschraenkung', $user->LicenseEffective->shortname, '-', true);
					}
				}
			}
		} else {
			if ($user->isActive()) {
				$currentYear = intval(date('Y'));
				$lastLicenseYear = $user->last_license_year;
				if ($lastLicenseYear == null || $lastLicenseYear < $currentYear) {
					$text = $lastLicenseYear == null ? \User::getStatusAsTextStatic(\User::STATE_UNLICENSED) . ' (letztes Lizenzjahr ?)' : \User::getStatusAsTextStatic(\User::STATE_UNLICENSED) . ' (letztes Lizenzjahr ' . $user->last_license_year . ')';
					$this->ausgeben = true;
					$this->abweichungen[] = new ESVAbweichung('person_status', $user->active, \User::STATE_UNLICENSED, true, \User::getStatusAsTextStatic($user->active), $text);
				}
			}
		}
	}

	public function isAusgeben()
	{
		return $this->ausgeben;
	}

	/**
	 * @return \User
	 */
	public function getUser()
	{
		return $this->user;
	}

	/**
	 * @return ESVAbweichung[]
	 */
	public function getAbweichungen()
	{
		return $this->abweichungen;
	}
}