<?php

namespace ZWE\Context;

/**
 * Abstrakte Basis-Klasse für ContextHolder, der eine Kontext-Variable in der
 * CodeIgniter-Session speichert. Diese Klasse benutzt das globale
 * CodeIgniter-Objekt und kann daher nur in CodeIgniter verwendet
 * werden.
 *
 * @author  Hendrik Heneke <hendrik@pixelwunder.net>
 * @since   2.5
 */
abstract class CISessionContextHolder implements ContextHolder {

    private $ci;
    private $contextVar;
    
    public function __construct() {
        // @phpstan-ignore-next-line
        $this->ci = & get_instance();
    }

    public function clearContext() {
        $this->contextVar = null;
        $this->ci->session->unset_userdata($this->getSessionKey());
    }

    public function getContext() {
        if($this->contextVar == null) {
            $this->contextVar = $this->retrieveFromContext();
        }
        return $this->contextVar;
    }

    /**
     * Gibt den Schlüssel für das Speichern in der CodeIgniter-Session
     * zurück.
     *
     * @return string
     */
    protected abstract function getSessionKey();

    /**
     * Gibt die Kontext-Variable zurück.
     *
     * @return mixed
     */
    protected abstract function retrieveFromContext();

    /**
     * Hilfsfunktion: Holt den zum in <code>getSessionKey</code> Schlüssel
     * gespeicherten Wert aus der CodeIgniter-Session.
     *
     * @return mixed
     */
    protected final function retrieveValueFromCISession() {
         return $this->ci->session->userdata($this->getSessionKey());
    }

    /**
     * Speichert die Kontext-Variable.
     *
     * @param mixed $contextVar Kontext-Variable
     */
    protected abstract function saveInContext($contextVar);

    /**
     * Hilfsfunktion: Speichert einen Wert in der CodeIgniter-Session unter
     * Verwendung des Schlüssels aus <code>getSessionKey</code>.
     *
     * @param mixed $value zu speichernder Wert
     */
    protected final function saveValueInCISession($value) {
        $this->ci->session->set_userdata($this->getSessionKey(), $value);
    }

    public function setContext($contextVar) {
        $this->contextVar = $contextVar;
        $this->saveInContext($contextVar);
    }

}