<?php

class DoctrineInitializer
{

	/**
	 * @var DoctrineInitializer
	 */
	private static $instance;

	/**
	 * @var Doctrine_Manager
	 */
	private $manager;

	/**
	 * @var string
	 */
	private $connectionString;

	/**
	 * @var Doctrine_Connection
	 */
	private $connection;

	private $sqlMode = "STRICT_TRANS_TABLES,NO_ZERO_IN_DATE,NO_ZERO_DATE,ERROR_FOR_DIVISION_BY_ZERO,NO_ENGINE_SUBSTITUTION";

	public function __construct($connectionString, array $modelDirectories = null)
	{
		if ($connectionString == null) {
			throw new InvalidArgumentException('Connection string may not be NULL!');
		}
		$this->connectionString = $connectionString;

		$this->manager = Doctrine_Manager::getInstance();
		$this->manager->setAttribute(Doctrine::ATTR_MODEL_LOADING, Doctrine::MODEL_LOADING_CONSERVATIVE);
		$this->manager->setAttribute(Doctrine::ATTR_AUTOLOAD_TABLE_CLASSES, true);
		$this->manager->setAttribute(Doctrine::ATTR_DEFAULT_TABLE_CHARSET, 'utf8');
		$this->manager->setAttribute(Doctrine::ATTR_DEFAULT_TABLE_COLLATE, 'utf8_general_ci');
		$this->manager->setAttribute(Doctrine::ATTR_DEFAULT_TABLE_TYPE, 'INNODB');

		spl_autoload_register(array('Doctrine', 'autoload'));

		if ($modelDirectories != null && count($modelDirectories) > 0) {
			Doctrine::loadModels($modelDirectories);
			spl_autoload_register(array('Doctrine', 'modelsAutoload'));
		}

		DoctrineInitializer::$instance = $this;
	}

	public static function getInstance()
	{
		if (self::$instance == null) {
			throw new \RuntimeException("No instance of DoctrineInitializer available!");
		}
		return self::$instance;
	}

	private function checkConnected()
	{
		if ($this->connection == null) {
			throw new RuntimeException("No connected!");
		}
	}

	private function checkDirectory($directory)
	{
		if ($directory == null) {
			throw new InvalidArgumentException("Migration directory is required!");
		}
		if (!is_dir($directory)) {
			throw new InvalidArgumentException("{$directory} is not a valid directory!");
		}
	}

	public function connect($checkVersion = null)
	{
		try {
			$this->connection = $this->manager->connection($this->connectionString);
			$this->connection->setCharset("utf8");
			$this->connection->getDbh()->query("SET SESSION sql_mode = '{$this->sqlMode}'");
			$this->afterConnected();
		} catch (Exception $ex) {
			if (!preg_match("/unknown database/i", $ex->getMessage())) {
				throw $ex;
			}
			$this->manager->getCurrentConnection()->createDatabase();
			$this->connection->setCharset("utf8");
			$this->connection->getDbh()->query("SET SESSION sql_mode = '{$this->sqlMode}'");
			$this->afterConnected();
		}

		if ($checkVersion != null) {
			$version = $this->getDatabaseVersion();
			if ($version != $checkVersion) {
				throw new RuntimeException("Database version mismatch: expected version {$checkVersion} found {$version}!");
			}
		}
	}

	protected function afterConnected() {
	}

	public function createStructureFromMigrations($directory, $version = null)
	{
		$this->checkConnected();
		$this->checkDirectory($directory);
		$this->connection->beginTransaction();
		$migration = new Doctrine_Migration($directory);
		if ($version == null) {
			$migration->migrate();
		} else {
			$migration->migrate($version);
		}
	}

	/**
	 * Creates all database tables using the loaded model classes
	 */
	public function createStructureFromModels()
	{
		$this->checkConnected();
		$this->connection->beginTransaction();
		Doctrine::createTablesFromModels();
	}

	public function disconnect()
	{
		if ($this->connection != null) {
			$this->beforeDisconnect();
			$this->manager->closeConnection($this->connection);
			$this->connection = null;
		}
	}

	protected function beforeDisconnect() {
	}

	/**
	 * Drops and creates the database
	 */
	public function dropCreateDatabase()
	{
		$this->checkConnected();
		$this->connection->dropDatabase();
		$this->connection->createDatabase();
	}

	/**
	 * Returns the current connection
	 *
	 * @return Doctrine_Connection
	 */
	public function getConnection()
	{
		$this->checkConnected();
		return $this->connection;
	}

	/**
	 * Returns the current database version
	 *
	 * @return int Database version or -1 if no table migration_version exists
	 */
	public function getDatabaseVersion()
	{
		$this->checkConnected();
		$tables = $this->connection->fetchColumn("SHOW tables");

		$found = false;
		foreach ($tables as $table) {
			if ($table == 'migration_version') {
				$found = true;
				break;
			}
		}

		if (!$found) {
			return -1;
		}

		$result = $this->connection->fetchColumn("SELECT version FROM migration_version");
		return isset($result[0]) ? $result[0] : 0;
	}

	public function loadFixtures($directory, $append = false)
	{
		$this->checkConnected();
		$this->checkDirectory($directory);
		Doctrine::loadData($directory, $append);
	}
}
