<?php
declare(strict_types=1);

namespace LTVBerlin\Contao\LTVBerlinBundle\Controller\FrontendModule;

use Contao\CoreBundle\Controller\FrontendModule\AbstractFrontendModuleController;
use Contao\FilesModel;
use Contao\StringUtil;
use LTVBerlin\Contao\LTVBerlinBundle\Model\BelegungsplanModel;

abstract class AbstractLLZFrontendModuleController extends AbstractFrontendModuleController
{
    protected const PLAN_REGEX = "/rvg(\d+)\_kw(\d{2})\.pdf/i";
    protected string $currentWeekNumber;
    protected string $currentYear;

    protected function __construct()
    {
        $this->currentWeekNumber = date('W');
        $this->currentYear = date('o');
    }

    protected function convertToPlan(FilesModel $model, string $year, string $weekNumber): BelegungsplanModel
    {
        return new BelegungsplanModel($model, $year, $weekNumber);
    }

    /**
     * @param $multiSRC array|string|null
     * @return BelegungsplanModel[]
     */
    protected function findPlans($multiSRC): array
    {
        if (!$multiSRC) {
            return [];
        }

        $objSources = FilesModel::findMultipleByIds(StringUtil::deserialize($multiSRC, true));
        if (!$objSources) {
            return [];
        }

        $arrFiles = [];

        while ($objSources->next()) {
            $objSource = $objSources->current();
            if ($objSource->type == 'folder') {
                $objFiles = FilesModel::findMultipleByBasepath($objSource->path);
                if ($objFiles) {
                    while ($objFiles->next()) {
                        $objFile = $objFiles->current();
                        if ($objFile->type == 'file' && $objFile->found) {
                            $arrFiles[] = $objFile;
                        }
                    }
                }
            } elseif ($objSource->type == 'file' && $objSource->found) {
                $arrFiles[] = $objSource;
            }
        }

        $arrFilteredPlans = [];
        foreach ($arrFiles as $objFile) {
            $matches = [];
            if (preg_match(self::PLAN_REGEX, $objFile->name, $matches)) {
                $year = $matches[1];
                $week = $matches[2];
                $objPlan = $this->convertToPlan($objFile, $year, $week);
                if ($year == $this->currentYear && $week >= $this->currentWeekNumber) {
                    $arrFilteredPlans[] = $objPlan;
                } elseif ($year > $this->currentYear) {
                    $arrFilteredPlans[] = $objPlan;
                }
            }
        }
        usort($arrFilteredPlans, array($this, 'sortPlans'));

        return $arrFilteredPlans;
    }

    protected function getFilenameForCurrentPlan(): string
    {
        return sprintf('rvg%1$s_kw%2$s.pdf', $this->currentYear, $this->currentWeekNumber);
    }

    protected function sortPlans(BelegungsplanModel $a, BelegungsplanModel $b): int
    {
        return strcmp("{$a->year}{$a->weekNumber}", "{$b->year}{$b->weekNumber}");
    }
}
