<?php
declare(strict_types=1);

namespace LTVBerlin\Contao\LTVBerlinBundle\Controller\BackendModule;

use Contao\BackendModule;
use Contao\Controller;
use Contao\FilesModel;
use Contao\FileTree;
use Contao\Input;
use Contao\Message;
use Contao\ZipReader;
use Contao\ZipWriter;

class UnzipBackendModule extends BackendModule
{

    protected $strTemplate = 'be_ltvberlin_unzip';

    protected function compile()
    {
        $error = false;
        $file = Input::get('id');
        $entries = [];

        if ($file) {
            $objModel = FilesModel::findByPath($file);
            if ($objModel) {
                try {
                    $extractTo = $this->getExtractTo($objModel);
                    $arc = $this->openZip($objModel);

                    if (Input::post('do_unzip') == 1) {
                        $success = $arc->extractTo($extractTo->absolutePath);
                        if ($success) {
                            Message::addConfirmation("Die Datei wurde in den Ordner {$extractTo->relativePath} entpackt.");
                        } else {
                            Message::addError("Die Datei konnte nicht entpackt werden.");
                        }
                    } else {
                        Message::addInfo("Die Datei wird voraussichtlich in den Ordner {$extractTo->relativePath} extrahiert.");

                        for ($i = 0; $i < $arc->numFiles; $i++) {
                            $entries[] = $arc->getNameIndex($i);
                        }
                        sort($entries);
                    }
                } catch (\Throwable $r) {
                    Message::addError("Fehler beim Einlesen der der Datei: {$r->getMessage()}");
                    $error = true;
                }
            } else {
                Message::addError("Datei {$file} nicht gefunden!");
                $error = true;
            }
        } else {
            Message::addError('Keine Datei ausgewählt!');
            $error = true;
        }

        $this->Template->messages = Message::generate();
        $this->Template->entries = $entries;
        $this->Template->href = $this->getReferer(true);
        $this->Template->title = !$error && $file ? $file : 'Fehler';
        $this->Template->button = $GLOBALS['TL_LANG']['MSC']['backBT'];
        $this->Template->action = ampersand(\Environment::get('request'));
        $this->Template->submitButton = specialchars($GLOBALS['TL_LANG']['MSC']['continue']);
    }

    private function getExtractTo(FilesModel $model): \stdClass
    {
        $absolutePath = $model->getAbsolutePath();
        $absoluteDirectory = dirname($absolutePath);
        $relatievDirectory = dirname($model->path);
        $folder = basename($absolutePath, '.zip');

        $result = $this->getNextUniqueFilename($absoluteDirectory, $folder);
        return (object) [
            'absolutePath' => $result->absolutePath,
            'relativePath' => "{$relatievDirectory}/{$result->folder}"
        ];
    }

    private function getNextUniqueFilename(string $directory, string $folder, int $depth = 0): \stdClass
    {
        $folderWithSuffix = $depth > 0 ? "{$folder}_$depth" : "{$folder}";
        $extractTo = "{$directory}/{$folderWithSuffix}";
        if (file_exists($extractTo)) {
            return $this->getNextUniqueFilename($directory, $folder, ++$depth);
        } else {
            return (object) [
                'absolutePath' => $extractTo,
                'folder' => $folderWithSuffix
            ];
        }
    }

    private function openZip(FilesModel $model): \ZipArchive
    {
        $arc = new \ZipArchive();
        $success = $arc->open($model->getAbsolutePath());
        if ($success !== true) {
            $message = $success !== false ? "ZIP-Datei konnte nicht geöffnet werden (Fehler-Code: {$success})." : "ZIP-Datei konnte nicht geöfnet werden (unbekannter Fehler).";
            throw new \RuntimeException($message);
        } else {
            return $arc;
        }
    }
}
