<?php
declare(strict_types=1);

namespace HHIT\VerbandVereinBundle\DCA\Callback;

use Contao\Controller;
use Contao\CoreBundle\Framework\ContaoFramework;
use Contao\CoreBundle\ServiceAnnotation\Callback;
use Contao\DataContainer;
use Contao\Input;
use Contao\MemberGroupModel;
use Contao\Message;
use Contao\StringUtil;
use Doctrine\DBAL\Connection;
use HHIT\VerbandVereinBundle\Models\VereinModel;

class VereinCallbacks
{

    private Connection $connection;
    private ContaoFramework $framework;

    public function __construct(Connection $connection, ContaoFramework $framework)
    {
        $this->connection = $connection;
        $this->framework = $framework;
        $this->framework->initialize(false);
    }

    /**
     * @Callback(
     *     table="tl_verband_verein",
     *     target="config.onsubmit"
     * )
     */
    public function onSubmit(DataContainer $dca)
    {
        $adapter = $this->framework->getAdapter(MemberGroupModel::class);
        /**
         * @var $memberGroup MemberGroupModel
         */
        $memberGroup = $adapter->findOneBy('verband_verein', $dca->activeRecord->id);
        if ($memberGroup == null) {
            $memberGroup = new MemberGroupModel();
            $memberGroup->verband_verein = $dca->activeRecord->id;
        }
        $memberGroup->name = VereinModel::createMemberGroupName($dca->activeRecord->verein);
        if ($memberGroup->isModified()) {
            $memberGroup->tstamp = time();
            $memberGroup->save();
        }
    }

    /**
     * @Callback(
     *     table="tl_verband_verein",
     *     target="config.ondelete"
     * )
     */
    public function onDelete(DataContainer $dca)
    {
        $adapter = $this->framework->getAdapter(MemberGroupModel::class);
        /**
         * @var $memberGroup MemberGroupModel | null
         */
        $memberGroup = $adapter->findOneBy('verband_verein', $dca->activeRecord->id);
        if ($memberGroup) {
            $memberGroup->delete();
        }
    }

    /**
     * @Callback(
     *     table="tl_verband_verein",
     *     target="fields.alias.save"
     * )
     */
    public function onSaveAlias($value, DataContainer $dc)
    {
        $autoGenerated = false;

        if (trim($value) == '') {
            $autoGenerated = true;
            $value = $this->generateAlias($dc->activeRecord->verein);
        }

        if ($this->aliasExists($value)) {
            if ($autoGenerated) {
                $value .= '-' . $dc->activeRecord->id;
            } else {
                throw new \Exception(sprintf($GLOBALS['TL_LANG']['ERR']['aliasExists'], $value));
            }
        }

        return $value;
    }

    /**
     * @Callback(
     *     table="tl_verband_verein",
     *     target="select.buttons"
     * )
     */
    public function onAddSelectButtons(array $arrButtons): array
    {
        /**
         * @var Input $adapter
         */
        $adapter = $this->framework->getAdapter(Input::class);
        $controller = $this->framework->getAdapter(Controller::class);

        if ($adapter->post('FORM_SUBMIT') == 'tl_select' && $adapter->post('generateAliases')) {
            $ids = $adapter->post('IDS');

            $models = $this->framework->getAdapter(VereinModel::class)->findMultipleByIds($ids);
            if ($models) {
                while ($models->next()) {
                    $verein = $models->current();
                    $verein->alias = $this->generateAlias($verein->verein);
                    if ($verein->isModified()) {
                        $verein->save();
                        Message::addInfo(sprintf('Alias für Verein <strong>%1$s</strong> generiert.', $verein->verein));
                    }
                }
            }

            $controller->redirect($controller->getReferer());
        }
        $arrButtons['verbandVereinAlias'] = '<button type="submit" name="generateAliases" id="generateAliases" class="tl_submit" accesskey="a" value="1">' . StringUtil::specialchars($GLOBALS['TL_LANG']['MSC']['aliasSelected']) . '</button>';
        return $arrButtons;
    }

    /**
     * @Callback(
     *     table="tl_verband_verein",
     *     target="list.global_operations.synchronizeMemberGroups.button"
     * )
     */
    public function synchronizeMemberGroups($href, $label, $title, $class, $attributes, $table, $ids)
    {
        $controller = $this->framework->getAdapter(Controller::class);
        return sprintf('<a href="%1$s" class="%2$s" %3$s title="%4$s" accesskey="s" onclick="Backend.getScrollOffset()">%4$s</a>', $controller->getReferer() . '&amp;' . $href, $class, $attributes, $label);
    }

    private function aliasExists(string $alias): bool
    {
        $result = $this->connection->prepare('SELECT id FROM tl_verband_verein WHERE alias = ?')->executeQuery([$alias]);
        return $result->rowCount() > 1;
    }

    private function generateAlias(string $name): string
    {
        return $this->postprocessAlias(StringUtil::generateAlias($name));
    }

    private function postprocessAlias(string $alias): string
    {
        return preg_replace(['/\-e\-?v$/', '/\-e\-?v\-/'], ['', '-'], $alias);
    }
}
