<?php
declare(strict_types=1);

namespace HHIT\VerbandVereinBundle\Controller\FrontendModule;

use Contao\Controller;
use Contao\CoreBundle\ServiceAnnotation\FrontendModule;
use Contao\Email;
use Contao\FrontendTemplate;
use Contao\FrontendUser;
use Contao\Idna;
use Contao\MemberGroupModel;
use Contao\Message;
use Contao\ModuleModel;
use Contao\RequestToken;
use Contao\StringUtil;
use Contao\Template;
use Contao\TextArea;
use Contao\TextField;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use HHIT\VerbandVereinBundle\Models\AktivitaetModel;
use HHIT\VerbandVereinBundle\Widgets\CheckboxGroup;
use HHIT\VerbandVereinBundle\Models\StandortModel;
use HHIT\VerbandVereinBundle\Models\VereinModel;

/**
 * @FrontendModule("verband_vereinsdateneditor",
 *  category="verband",
 *  template="mod_verband_verein_editor",
 *  renderer="forward"
 * )
 */
class VereinsdateneditorFrontendModuleController extends AbstractVereinFrontendModuleController
{
    protected function getResponse(Template $template, ModuleModel $model, Request $request): ?Response
    {
        $requestedVereinsId = $request->query->get('v');

        $frontendUser = $this->checkAuthenticated();
        $allowedVereine = $this->loadAllowedVereine($frontendUser);

        if ($requestedVereinsId) {
            $requestedVerein = null;
            foreach ($allowedVereine as $allowedVerein) {
                if ($allowedVerein->id == $requestedVereinsId) {
                    $requestedVerein = $allowedVerein;
                }
            }
            if (!$requestedVerein) {
                throw $this->createAccessDeniedException();
            }
            $response = $this->compileEditor($requestedVerein, $model, $frontendUser, $request);
            if ($response instanceof RedirectResponse) {
                return $response;
            } else {
                $template->contents = $response;
            }
        } else {
            $template->contents = $this->compileList($allowedVereine);
        }

        return $template->getResponse();
    }

    private function checkAuthenticated(): FrontendUser
    {
        $adapter = $this->framework->getAdapter(FrontendUser::class);
        $user = $adapter->getInstance();
        if (!$user) {
            throw $this->createAccessDeniedException();
        }
        return $user;
    }

    /**
     * @param FrontendUser $user
     * @return VereinModel[]
     */
    private function loadAllowedVereine(FrontendUser $user): array
    {
        $vereine = [];

        $adapterMemberGroup = $this->framework->getAdapter(MemberGroupModel::class);
        $groupCollection = $adapterMemberGroup->findMultipleByIds($user->groups);

        $vereinsIds = [];
        if ($groupCollection) {
            while ($groupCollection->next()) {
                $memberGroup = $groupCollection->current();
                if ($memberGroup->verband_verein > 0) {
                    $vereinsIds[] = $memberGroup->verband_verein;
                }
            }
        }

        $adapterVereinModel = $this->framework->getAdapter(VereinModel::class);
        $vereinCollection = $adapterVereinModel->findMultipleByIds($vereinsIds);

        if ($vereinCollection) {
            while ($vereinCollection->next()) {
                $vereine[] = $vereinCollection->current();
            }
        }

        return $vereine;
    }

    private function compileEditor(VereinModel $verein, ModuleModel $model, FrontendUser $user, Request $request)
    {
        $GLOBALS['TL_JAVASCRIPT'][] = 'bundles/verbandverein/ckeditor/4.2/ckeditor.js';
        $adapter = $this->framework->getAdapter(Controller::class);
        $adapter->loadLanguageFile('tl_verband_verein');
        $adapter->loadDataContainer('tl_verband_verein');

        $template = new FrontendTemplate('vde_editor');
        $template->verein = $verein;

        $fieldsets = $this->createEmptyFieldsets();

        $evals = array('rgxp', 'maxlength', 'mandatory', 'allowHtml', 'multiple');
        foreach ($fieldsets as $fieldset) {
            foreach ($fieldset->fields as $field) {

                $widget = null;
                switch ($GLOBALS['TL_DCA']['tl_verband_verein']['fields'][$field]['inputType']) {
                    case 'textarea':
                        $widget = new TextArea();
                        $widget->class = "textarea";
                        $widget->rows = 6;
                        break;
                    case 'checkboxWizard':
                        $widget = new CheckboxGroup();
                        $widget->options = $this->loadCheckboxGroupOptions($field);
                        break;
                    default:
                        $widget = new TextField();
                        $widget->class = "text";
                }
                $widget->id = $field;
                $widget->name = $field;
                $widget->label = $GLOBALS['TL_DCA']['tl_verband_verein']['fields'][$field]['label'][0];
                foreach ($evals as $eval) {
                    if (isset($GLOBALS['TL_DCA']['tl_verband_verein']['fields'][$field]['eval'][$eval])) {
                        $widget->$eval = $GLOBALS['TL_DCA']['tl_verband_verein']['fields'][$field]['eval'][$eval];
                        if ($eval == 'allowHtml') {
                            $widget->class .= " rte";
                            $widget->addAttribute("decodeEntities", true);
                        }
                    }
                }
                $widget->value = $verein->$field;
                $fieldset->widgets[] = $widget;
            }
        }

        $valid = true;
        foreach ($fieldsets as $fieldset) {
            foreach ($fieldset->widgets as $widget) {
                $renderedWidget = new \stdClass();
                $renderedWidget->label = $widget->generateLabel();
                if ($_POST) {
                    $widget->validate();
                    if ($widget->hasErrors()) {
                        $renderedWidget->widget = $this->removeBackendJs($widget->generateWithError());
                        $valid = false;
                    } else {
                        $renderedWidget->widget = $this->removeBackendJs($widget->generate());
                    }
                } else {
                    $renderedWidget->widget = $this->removeBackendJs($widget->generate());
                }
                $fieldset->renderedWidgets[] = $renderedWidget;
            }
        }
        $template->fieldsets = $fieldsets;
        $template->token = RequestToken::get();

        if ($request->isMethod('post') && $valid) {
            $changed = false;
            $changedProperties = [];
            foreach ($fieldsets as $fieldset) {
                foreach ($fieldset->widgets as $widget) {
                    $property = $widget->name;
                    $originalValue = StringUtil::deserialize($verein->$property);
                    if (is_array($originalValue)) {
                        if (count(array_diff($widget->value, $originalValue)) > 0 || count(array_diff($originalValue, $widget->value)) > 0) {
                            $verein->$property = $widget->value;
                            $changed = true;
                            $changedProperties[] = $property;
                        }
                    } else {
                        if ($originalValue != $widget->value) {
                            $verein->$property = $widget->value;
                            $changed = true;
                            $changedProperties[] = $property;
                        }
                    }
                }
            }
            if ($changed) {
                $verein->save();

                $emailTemplate = new FrontendTemplate('vde_email');
                $emailTemplate->arrProps = array('adresse', 'telefon', 'fax', 'email', 'homepage', 'standorte', 'trainingsstaetten', 'aktivitaeten', 'weitere_aktivitaeten', 'ansprechpartner');
                $emailTemplate->arrChangedProps = $changedProperties;
                $emailTemplate->verein = $verein;
                $emailTemplate->labels = $GLOBALS['TL_DCA']['tl_verband_verein']['fields'];
                $emailTemplate->objMember = $user;

                if ($model->verband_recipient) {
                    $email = new Email();
                    $email->from = $GLOBALS['TL_ADMIN_EMAIL'];
                    $email->fromName = $GLOBALS['TL_ADMIN_NAME'];
                    $email->subject = "Vereinsdatenänderung " . $verein->verein;
                    $email->html = $emailTemplate->parse();
                    $email->sendTo(Idna::encodeEmail($model->verband_recipient));
                }

                Message::addConfirmation("Vereinsdaten aktualisiert.");
            } else {
                Message::addInfo("Sie haben keine Änderungen vorgenommen - Datensatz nicht aktualisiert.");
            }

            return $this->redirect(Controller::getReferer());
        }

        $template->message = Message::generate();
        Message::reset();

        return $template->parse();
    }

    /**
     * @param VereinModel[] $vereine
     * @return string
     */
    private function compileList(array $vereine): string
    {
        $template = new FrontendTemplate('vde_liste');
        $template->baseUri = $this->getPageModel()->getFrontendUrl();
        $template->vereine = $vereine;
        return $template->parse();
    }

    private function createEmptyFieldsets(): array
    {
        $fieldsets = [];

        $fieldset = new \stdClass();
        $fieldset->legend = "Stammdaten";
        $fieldset->fields = ['adresse', 'telefon', 'fax', 'email', 'homepage'];
        $fieldset->widgets = [];
        $fieldset->renderedWidgets = [];
        $fieldsets[] = $fieldset;

        $fieldset = new \stdClass();
        $fieldset->legend = "Standorte";
        $fieldset->fields = ["standorte", "trainingsstaetten"];
        $fieldset->widgets = [];
        $fieldset->renderedWidgets = [];
        $fieldsets[] = $fieldset;

        $fieldset = new \stdClass();
        $fieldset->legend = "Aktivitäten";
        $fieldset->fields = ["aktivitaeten", "weitere_aktivitaeten"];
        $fieldset->widgets = [];
        $fieldset->renderedWidgets = [];
        $fieldsets[] = $fieldset;

        $fieldset = new \stdClass();
        $fieldset->legend = "Ansprechpartner";
        $fieldset->fields = ["ansprechpartner"];
        $fieldset->widgets = [];
        $fieldset->renderedWidgets = [];
        $fieldsets[] = $fieldset;

        return $fieldsets;
    }

    protected function loadCheckboxGroupOptions($field): array
    {
        $recordCollection = null;
        $property = null;

        switch ($field) {
            case "standorte":
                $recordCollection = StandortModel::findAll();
                $property = "standort";
                break;
            case "aktivitaeten":
                $recordCollection = AktivitaetModel::findAll();
                $property = "aktivitaet";
                break;
        }

        $options = [];

        if ($recordCollection) {
            while ($recordCollection->next()) {
                $record = $recordCollection->current();
                if ($property) {
                    $option = new \stdClass();
                    $option->value = $record->id;
                    $option->label = $record->$property;
                    $options[] = $option;
                }
            }
        }

        return $options;
    }

    private function removeBackendJs(string $string): string
    {
        return str_replace('onfocus="Backend.getScrollOffset()"', '', $string);
    }
}
